# database.py - مدیریت اتصال به دیتابیس و ایجاد جداول

import aiosqlite
from config import DATABASE_PATH


class Database:
    """کلاس مدیریت اتصال به دیتابیس SQLite با پشتیبانی async."""
    
    _instance = None
    _connection = None

    @classmethod
    async def get_connection(cls) -> aiosqlite.Connection:
        """دریافت اتصال به دیتابیس (Singleton Pattern)."""
        if cls._connection is None:
            cls._connection = await aiosqlite.connect(DATABASE_PATH)
            cls._connection.row_factory = aiosqlite.Row
            await cls._create_tables()
        return cls._connection

    @classmethod
    async def _create_tables(cls):
        """ایجاد جداول مورد نیاز در صورت عدم وجود."""
        conn = cls._connection
        
        # جدول کاربران
        await conn.execute("""
            CREATE TABLE IF NOT EXISTS users (
                telegram_id INTEGER PRIMARY KEY,
                phone_number TEXT UNIQUE,
                full_name TEXT,
                field TEXT,
                grade TEXT,
                province TEXT,
                is_active INTEGER DEFAULT 1,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
            )
        """)
        
        # جدول سوالات (مخزن A)
        await conn.execute("""
            CREATE TABLE IF NOT EXISTS questions (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                text TEXT NOT NULL,
                option_a TEXT NOT NULL,
                option_b TEXT NOT NULL,
                option_c TEXT NOT NULL,
                option_d TEXT NOT NULL,
                correct_answer TEXT NOT NULL CHECK(correct_answer IN ('A', 'B', 'C', 'D')),
                explanation TEXT,
                category TEXT,
                difficulty TEXT DEFAULT 'medium',
                is_active INTEGER DEFAULT 1,
                created_by INTEGER,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
            )
        """)
        
        # جدول نتایج آزمون
        await conn.execute("""
            CREATE TABLE IF NOT EXISTS exam_results (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                user_id INTEGER NOT NULL,
                total_questions INTEGER NOT NULL,
                correct_answers INTEGER NOT NULL,
                score REAL NOT NULL,
                exam_date TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                FOREIGN KEY (user_id) REFERENCES users(telegram_id)
            )
        """)
        
        # جدول سوالات در انتظار تایید (ارسال شده توسط کاربران)
        await conn.execute("""
            CREATE TABLE IF NOT EXISTS pending_questions (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                text TEXT NOT NULL,
                option_a TEXT NOT NULL,
                option_b TEXT NOT NULL,
                option_c TEXT NOT NULL,
                option_d TEXT NOT NULL,
                correct_answer TEXT NOT NULL CHECK(correct_answer IN ('A', 'B', 'C', 'D')),
                explanation TEXT,
                submitted_by INTEGER NOT NULL,
                submitted_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                status TEXT DEFAULT 'pending' CHECK(status IN ('pending', 'approved', 'rejected')),
                reviewed_by INTEGER,
                reviewed_at TIMESTAMP,
                FOREIGN KEY (submitted_by) REFERENCES users(telegram_id)
            )
        """)
        
        # جدول سوالات تایید شده کاربران (مخزن B - برای تست شانسی)
        await conn.execute("""
            CREATE TABLE IF NOT EXISTS user_questions (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                text TEXT NOT NULL,
                option_a TEXT NOT NULL,
                option_b TEXT NOT NULL,
                option_c TEXT NOT NULL,
                option_d TEXT NOT NULL,
                correct_answer TEXT NOT NULL CHECK(correct_answer IN ('A', 'B', 'C', 'D')),
                explanation TEXT,
                submitted_by INTEGER NOT NULL,
                is_active INTEGER DEFAULT 1,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                FOREIGN KEY (submitted_by) REFERENCES users(telegram_id)
            )
        """)
        
        # جدول پیش‌آزمون‌ها
        await conn.execute("""
            CREATE TABLE IF NOT EXISTS preexams (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                title TEXT NOT NULL,
                description TEXT,
                is_active INTEGER DEFAULT 1,
                order_num INTEGER DEFAULT 0,
                created_by INTEGER,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
            )
        """)
        
        # جدول ارتباط سوالات با پیش‌آزمون (سوالات ثابت هر پیش‌آزمون)
        await conn.execute("""
            CREATE TABLE IF NOT EXISTS preexam_questions (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                preexam_id INTEGER NOT NULL,
                question_id INTEGER NOT NULL,
                question_order INTEGER DEFAULT 0,
                FOREIGN KEY (preexam_id) REFERENCES preexams(id),
                FOREIGN KEY (question_id) REFERENCES questions(id),
                UNIQUE(preexam_id, question_id)
            )
        """)
        
        # جدول نتایج پیش‌آزمون کاربران
        await conn.execute("""
            CREATE TABLE IF NOT EXISTS preexam_results (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                user_id INTEGER NOT NULL,
                preexam_id INTEGER NOT NULL,
                total_questions INTEGER NOT NULL,
                correct_answers INTEGER NOT NULL,
                wrong_answers INTEGER NOT NULL,
                blank_answers INTEGER NOT NULL,
                score_simple REAL NOT NULL,
                score_negative REAL NOT NULL,
                answers_json TEXT,
                exam_date TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                FOREIGN KEY (user_id) REFERENCES users(telegram_id),
                FOREIGN KEY (preexam_id) REFERENCES preexams(id)
            )
        """)
        
        # جدول پیام‌های ناشناس
        await conn.execute("""
            CREATE TABLE IF NOT EXISTS anonymous_messages (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                sender_id INTEGER NOT NULL,
                message_text TEXT NOT NULL,
                reply_text TEXT,
                replied_by INTEGER,
                is_read INTEGER DEFAULT 0,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                replied_at TIMESTAMP,
                FOREIGN KEY (sender_id) REFERENCES users(telegram_id)
            )
        """)
        
        # جدول درخواست‌های خدمات
        await conn.execute("""
            CREATE TABLE IF NOT EXISTS service_requests (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                user_id INTEGER NOT NULL,
                phone TEXT NOT NULL,
                service_type TEXT NOT NULL,
                status TEXT DEFAULT 'pending',
                receipt_file_id TEXT,
                is_approved INTEGER DEFAULT 0,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                FOREIGN KEY (user_id) REFERENCES users(telegram_id)
            )
        """)

        
        # جدول نظرات و امتیازات
        await conn.execute("""
            CREATE TABLE IF NOT EXISTS test_ratings (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                user_id INTEGER NOT NULL,
                question_id INTEGER NOT NULL,
                score INTEGER NOT NULL CHECK(score BETWEEN 1 AND 5),
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                FOREIGN KEY (user_id) REFERENCES users(telegram_id),
                FOREIGN KEY (question_id) REFERENCES user_questions(id),
                UNIQUE(user_id, question_id)
            )
        """)

        # اضافه کردن ستون‌های امتیازدهی به جدول user_questions (مایگریشن دستی)
        try:
            await conn.execute("ALTER TABLE user_questions ADD COLUMN rating_sum INTEGER DEFAULT 0")
        except Exception:
            pass  # ستون احتمالا وجود دارد
            
        try:
            await conn.execute("ALTER TABLE user_questions ADD COLUMN rating_count INTEGER DEFAULT 0")
        except Exception:
            pass
        
        # جدول تاریخچه تست شانسی (برای جلوگیری از تکرار)
        await conn.execute("""
            CREATE TABLE IF NOT EXISTS user_lucky_history (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                user_id INTEGER NOT NULL,
                question_id INTEGER NOT NULL,
                seen_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                FOREIGN KEY (user_id) REFERENCES users(telegram_id),
                FOREIGN KEY (question_id) REFERENCES user_questions(id),
                UNIQUE(user_id, question_id)
            )
        """)
        
        # جدول تنظیمات (برای ذخیره قیمت‌ها و...)
        await conn.execute("""
            CREATE TABLE IF NOT EXISTS settings (
                key TEXT PRIMARY KEY,
                value TEXT NOT NULL,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
            )
        """)
        
        # مقادیر پیش‌فرض قیمت‌ها
        await conn.execute("""
            INSERT OR IGNORE INTO settings (key, value) VALUES ('exam_price', '800000')
        """)
        await conn.execute("""
            INSERT OR IGNORE INTO settings (key, value) VALUES ('consult_price', '850000')
        """)
        
        await conn.commit()


    @classmethod
    async def close(cls):
        """بستن اتصال به دیتابیس."""
        if cls._connection:
            await cls._connection.close()
            cls._connection = None


async def init_database():
    """تابع راه‌اندازی اولیه دیتابیس."""
    await Database.get_connection()
    print("✅ دیتابیس با موفقیت راه‌اندازی شد (شامل جداول جدید).")
