# handlers/admin.py - پنل ادمین برای مدیریت سوالات

from telegram import (
    Update,
    InlineKeyboardButton,
    InlineKeyboardMarkup,
    ReplyKeyboardMarkup,
    ReplyKeyboardRemove,
)
from telegram.ext import (
    ContextTypes,
    ConversationHandler,
    CommandHandler,
    MessageHandler,
    CallbackQueryHandler,
    filters,
)

from config import ADMIN_IDS
from repositories import QuestionRepository, PendingQuestionRepository, UserQuestionRepository, PreexamRepository, ServiceRequestRepository, SettingsRepository, UserRepository


# ==================== State‌های ConversationHandler ====================
(
    QUESTION_TEXT,
    OPTION_A,
    OPTION_B,
    OPTION_C,
    OPTION_D,
    CORRECT_ANSWER,
    EXPLANATION,
    REPLY_TEXT,
    SELECT_PREEXAM,
    POST_MESSAGE_TEXT,
    POST_BUTTON_TEXT,
    POST_SELECT_PREEXAM,
    BROADCAST_MESSAGE,
    BROADCAST_CONFIRM,
    ABOUT_CONTENT,
) = range(15)







# ==================== دکوریتور چک ادمین ====================

def admin_only(func):
    """دکوریتور برای محدود کردن دسترسی به ادمین‌ها."""
    async def wrapper(update: Update, context: ContextTypes.DEFAULT_TYPE, *args, **kwargs):
        user_id = update.effective_user.id
        if user_id not in ADMIN_IDS:
            if update.callback_query:
                await update.callback_query.answer("❌ شما دسترسی ادمین ندارید!", show_alert=True)
            else:
                await update.message.reply_text("❌ شما دسترسی ادمین ندارید!")
            return ConversationHandler.END
        return await func(update, context, *args, **kwargs)
    return wrapper


# ==================== کیبوردها ====================

def get_admin_menu_keyboard() -> InlineKeyboardMarkup:
    """کیبورد منوی اصلی ادمین - ۵ بخش."""
    keyboard = [
        [InlineKeyboardButton("📚 مدیریت آزمون", callback_data="admin_exam_mgmt")],
        [InlineKeyboardButton("🛒 مدیریت ثبت‌نام محصولات", callback_data="admin_products_mgmt")],
        [InlineKeyboardButton("📨 مدیریت پیام‌های ناشناس", callback_data="admin_anon_messages")],
        [InlineKeyboardButton("📢 پیام همگانی", callback_data="admin_broadcast")],
        [InlineKeyboardButton("ℹ️ تنظیم معرفی مجموعه", callback_data="admin_set_about")],
        [InlineKeyboardButton("🔙 بازگشت", callback_data="admin_back")],
    ]
    return InlineKeyboardMarkup(keyboard)




def get_exam_management_keyboard() -> InlineKeyboardMarkup:
    """کیبورد مدیریت آزمون."""
    keyboard = [
        [InlineKeyboardButton("➕ افزودن سوال", callback_data="admin_add_question")],
        [InlineKeyboardButton("📥 سوالات در انتظار تایید", callback_data="admin_pending")],
        [InlineKeyboardButton("📋 مدیریت پیش‌آزمون‌ها", callback_data="admin_preexams")],
        [InlineKeyboardButton("📢 ارسال پست به کانال", callback_data="admin_channel_post")],
        [InlineKeyboardButton("📦 مخزن سوالات آزمون", callback_data="admin_questions_repo")],
        [InlineKeyboardButton("🎲 مخزن تست شانسی", callback_data="admin_lucky_repo")],
        [InlineKeyboardButton("📊 آمار سوالات", callback_data="admin_stats")],
        [InlineKeyboardButton("🔙 بازگشت به منوی اصلی", callback_data="back_to_admin_menu")],
    ]
    return InlineKeyboardMarkup(keyboard)



def get_products_management_keyboard() -> InlineKeyboardMarkup:
    """کیبورد مدیریت ثبت‌نام محصولات."""
    keyboard = [
        [InlineKeyboardButton("💰 تنظیم قیمت‌ها", callback_data="admin_price_settings")],
        [InlineKeyboardButton("📋 رسیدهای در انتظار بررسی", callback_data="admin_registrants")],
        [InlineKeyboardButton("✅ رسیدهای تایید شده", callback_data="admin_approved")],
        [InlineKeyboardButton("🔙 بازگشت به منوی اصلی", callback_data="back_to_admin_menu")],
    ]
    return InlineKeyboardMarkup(keyboard)


def get_price_settings_keyboard() -> InlineKeyboardMarkup:
    """کیبورد تنظیم قیمت‌ها."""
    keyboard = [
        [InlineKeyboardButton("💳 قیمت پکیج آزمون", callback_data="admin_set_exam_price")],
        [InlineKeyboardButton("📞 قیمت مشاوره تحصیلی", callback_data="admin_set_consult_price")],
        [InlineKeyboardButton("🔙 بازگشت", callback_data="admin_products_mgmt")],
    ]
    return InlineKeyboardMarkup(keyboard)


def get_correct_answer_keyboard() -> InlineKeyboardMarkup:
    """کیبورد انتخاب گزینه صحیح."""
    keyboard = [
        [
            InlineKeyboardButton("الف (A)", callback_data="correct_A"),
            InlineKeyboardButton("ب (B)", callback_data="correct_B"),
        ],
        [
            InlineKeyboardButton("ج (C)", callback_data="correct_C"),
            InlineKeyboardButton("د (D)", callback_data="correct_D"),
        ],
        [InlineKeyboardButton("❌ انصراف", callback_data="admin_cancel")],
    ]
    return InlineKeyboardMarkup(keyboard)


def get_cancel_keyboard() -> ReplyKeyboardMarkup:
    """کیبورد انصراف."""
    return ReplyKeyboardMarkup(
        [["❌ انصراف"]],
        resize_keyboard=True,
        one_time_keyboard=True
    )


def get_skip_keyboard() -> ReplyKeyboardMarkup:
    """کیبورد رد کردن (برای فیلدهای اختیاری)."""
    return ReplyKeyboardMarkup(
        [["⏭ رد کردن"], ["❌ انصراف"]],
        resize_keyboard=True,
        one_time_keyboard=True
    )


# ==================== هندلرها ====================

@admin_only
async def admin_command(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """هندلر دستور /admin - نمایش پنل ادمین."""
    question_count = await QuestionRepository.get_count()
    pending_count = await PendingQuestionRepository.get_pending_count()
    user_question_count = await UserQuestionRepository.get_count()
    pending_receipts = await ServiceRequestRepository.get_pending()
    
    await update.message.reply_text(
        f"🔐 *پنل مدیریت*\n\n"
        f"📝 سوالات آزمون: {question_count}\n"
        f"📥 در انتظار تایید: {pending_count}\n"
        f"🎲 تست شانسی: {user_question_count}\n"
        f"📋 رسیدهای در انتظار: {len(pending_receipts)}\n\n"
        f"از منوی زیر انتخاب کنید:",
        parse_mode="Markdown",
        reply_markup=get_admin_menu_keyboard()
    )


@admin_only
async def show_exam_management(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """نمایش منوی مدیریت آزمون."""
    query = update.callback_query
    await query.answer()
    
    question_count = await QuestionRepository.get_count()
    pending_count = await PendingQuestionRepository.get_pending_count()
    user_question_count = await UserQuestionRepository.get_count()
    
    await query.message.edit_text(
        f"📚 *مدیریت آزمون*\n\n"
        f"📝 سوالات مخزن A (آزمون): {question_count}\n"
        f"📥 سوالات در انتظار تایید: {pending_count}\n"
        f"🎲 سوالات مخزن B (شانسی): {user_question_count}\n\n"
        f"یک گزینه انتخاب کنید:",
        parse_mode="Markdown",
        reply_markup=get_exam_management_keyboard()
    )


@admin_only
async def show_products_management(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """نمایش منوی مدیریت ثبت‌نام محصولات."""
    query = update.callback_query
    await query.answer()
    
    pending_receipts = await ServiceRequestRepository.get_pending()
    
    await query.message.edit_text(
        f"🛒 *مدیریت ثبت‌نام محصولات*\n\n"
        f"📋 رسیدهای در انتظار بررسی: {len(pending_receipts)}\n\n"
        f"یک گزینه انتخاب کنید:",
        parse_mode="Markdown",
        reply_markup=get_products_management_keyboard()
    )


@admin_only
async def show_price_settings(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """نمایش تنظیمات قیمت."""
    query = update.callback_query
    await query.answer()
    
    # TODO: دریافت قیمت‌ها از دیتابیس (فعلاً ثابت)
    exam_price = "800,000"
    consult_price = "850,000"
    
    await query.message.edit_text(
        f"💰 *تنظیم قیمت‌ها*\n\n"
        f"💳 قیمت پکیج آزمون: {exam_price} تومان\n"
        f"📞 قیمت مشاوره تحصیلی: {consult_price} تومان\n\n"
        f"برای تغییر قیمت، روی گزینه مورد نظر کلیک کنید:",
        parse_mode="Markdown",
        reply_markup=get_price_settings_keyboard()
    )


@admin_only
async def start_set_exam_price(update: Update, context: ContextTypes.DEFAULT_TYPE) -> int:
    """شروع تنظیم قیمت آزمون."""
    query = update.callback_query
    await query.answer()
    
    context.user_data["price_type"] = "exam"
    
    await query.message.edit_text(
        "💳 *تنظیم قیمت پکیج آزمون*\n\n"
        "لطفاً قیمت جدید را به تومان وارد کنید:\n"
        "(فقط عدد، مثال: 800000)",
        parse_mode="Markdown"
    )
    await context.bot.send_message(
        chat_id=update.effective_chat.id,
        text="قیمت جدید:",
        reply_markup=get_cancel_keyboard()
    )
    return REPLY_TEXT


@admin_only
async def start_set_consult_price(update: Update, context: ContextTypes.DEFAULT_TYPE) -> int:
    """شروع تنظیم قیمت مشاوره."""
    query = update.callback_query
    await query.answer()
    
    context.user_data["price_type"] = "consult"
    
    await query.message.edit_text(
        "📞 *تنظیم قیمت مشاوره تحصیلی*\n\n"
        "لطفاً قیمت جدید را به تومان وارد کنید:\n"
        "(فقط عدد، مثال: 850000)",
        parse_mode="Markdown"
    )
    await context.bot.send_message(
        chat_id=update.effective_chat.id,
        text="قیمت جدید:",
        reply_markup=get_cancel_keyboard()
    )
    return REPLY_TEXT


async def receive_new_price(update: Update, context: ContextTypes.DEFAULT_TYPE) -> int:
    """دریافت قیمت جدید."""
    text = update.message.text.strip()
    
    if text == "❌ انصراف":
        await update.message.reply_text(
            "❌ تنظیم قیمت لغو شد.\n\nبرای بازگشت به پنل ادمین /admin را ارسال کنید.",
            reply_markup=ReplyKeyboardRemove()
        )
        return ConversationHandler.END
    
    # اعتبارسنجی عدد
    try:
        price = int(text.replace(",", "").replace("،", ""))
    except ValueError:
        await update.message.reply_text(
            "❌ لطفاً فقط عدد وارد کنید (مثال: 800000):"
        )
        return REPLY_TEXT
    
    price_type = context.user_data.get("price_type", "exam")
    
    # ذخیره در دیتابیس
    await SettingsRepository.set(f"{price_type}_price", str(price))
    
    price_formatted = f"{price:,}".replace(",", "،")
    price_label = "پکیج آزمون" if price_type == "exam" else "مشاوره تحصیلی"
    
    await update.message.reply_text(
        f"✅ قیمت {price_label} به {price_formatted} تومان تغییر یافت.\n\n"
        f"⚠️ توجه: این قیمت در نمایش به کاربران اعمال می‌شود.\n\n"
        f"برای بازگشت به پنل ادمین /admin را ارسال کنید.",
        reply_markup=ReplyKeyboardRemove()
    )
    return ConversationHandler.END


@admin_only
async def admin_stats_callback(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """نمایش آمار سوالات."""
    query = update.callback_query
    await query.answer()
    
    question_count = await QuestionRepository.get_count()
    pending_count = await PendingQuestionRepository.get_pending_count()
    user_question_count = await UserQuestionRepository.get_count()
    
    await query.message.edit_text(
        f"📊 *آمار مخزن سوالات*\n\n"
        f"📝 سوالات مخزن A (آزمون): {question_count}\n"
        f"📥 سوالات در انتظار تایید: {pending_count}\n"
        f"🎲 سوالات مخزن B (شانسی): {user_question_count}\n\n"
        f"برای افزودن سوال جدید از دکمه زیر استفاده کنید:",
        parse_mode="Markdown",
        reply_markup=get_admin_menu_keyboard()
    )


@admin_only
async def start_add_question(update: Update, context: ContextTypes.DEFAULT_TYPE) -> int:
    """شروع فرآیند افزودن سوال."""
    query = update.callback_query
    await query.answer()
    
    await query.message.edit_text(
        "📝 *افزودن سوال جدید - مرحله ۱ از ۷*\n\n"
        "لطفاً متن سوال را وارد کنید:",
        parse_mode="Markdown"
    )
    
    await query.message.reply_text(
        "در هر مرحله می‌توانید با ارسال «❌ انصراف» فرآیند را لغو کنید.",
        reply_markup=get_cancel_keyboard()
    )
    
    return QUESTION_TEXT


async def receive_question_text(update: Update, context: ContextTypes.DEFAULT_TYPE) -> int:
    """دریافت متن سوال."""
    text = update.message.text.strip()
    
    if text == "❌ انصراف":
        return await cancel_add_question(update, context)
    
    if len(text) < 10:
        await update.message.reply_text(
            "❌ متن سوال باید حداقل ۱۰ کاراکتر باشد. دوباره تلاش کنید:"
        )
        return QUESTION_TEXT
    
    # ذخیره موقت
    context.user_data["new_question"] = {"text": text}
    
    await update.message.reply_text(
        "📝 *مرحله ۲ از ۷*\n\n"
        "گزینه الف را وارد کنید:",
        parse_mode="Markdown",
        reply_markup=get_cancel_keyboard()
    )
    return OPTION_A


async def receive_option_a(update: Update, context: ContextTypes.DEFAULT_TYPE) -> int:
    """دریافت گزینه الف."""
    text = update.message.text.strip()
    
    if text == "❌ انصراف":
        return await cancel_add_question(update, context)
    
    context.user_data["new_question"]["option_a"] = text
    
    await update.message.reply_text(
        "📝 *مرحله ۳ از ۷*\n\n"
        "گزینه ب را وارد کنید:",
        parse_mode="Markdown",
        reply_markup=get_cancel_keyboard()
    )
    return OPTION_B


async def receive_option_b(update: Update, context: ContextTypes.DEFAULT_TYPE) -> int:
    """دریافت گزینه ب."""
    text = update.message.text.strip()
    
    if text == "❌ انصراف":
        return await cancel_add_question(update, context)
    
    context.user_data["new_question"]["option_b"] = text
    
    await update.message.reply_text(
        "📝 *مرحله ۴ از ۷*\n\n"
        "گزینه ج را وارد کنید:",
        parse_mode="Markdown",
        reply_markup=get_cancel_keyboard()
    )
    return OPTION_C


async def receive_option_c(update: Update, context: ContextTypes.DEFAULT_TYPE) -> int:
    """دریافت گزینه ج."""
    text = update.message.text.strip()
    
    if text == "❌ انصراف":
        return await cancel_add_question(update, context)
    
    context.user_data["new_question"]["option_c"] = text
    
    await update.message.reply_text(
        "📝 *مرحله ۵ از ۷*\n\n"
        "گزینه د را وارد کنید:",
        parse_mode="Markdown",
        reply_markup=get_cancel_keyboard()
    )
    return OPTION_D


async def receive_option_d(update: Update, context: ContextTypes.DEFAULT_TYPE) -> int:
    """دریافت گزینه د."""
    text = update.message.text.strip()
    
    if text == "❌ انصراف":
        return await cancel_add_question(update, context)
    
    context.user_data["new_question"]["option_d"] = text
    
    # نمایش خلاصه سوال و درخواست گزینه صحیح
    q = context.user_data["new_question"]
    summary = (
        f"📝 *مرحله ۶ از ۷ - انتخاب گزینه صحیح*\n\n"
        f"*سوال:*\n{q['text']}\n\n"
        f"*الف)* {q['option_a']}\n"
        f"*ب)* {q['option_b']}\n"
        f"*ج)* {q['option_c']}\n"
        f"*د)* {q['option_d']}\n\n"
        f"گزینه صحیح را انتخاب کنید:"
    )
    
    await update.message.reply_text(
        summary,
        parse_mode="Markdown",
        reply_markup=get_correct_answer_keyboard()
    )
    return CORRECT_ANSWER


async def receive_correct_answer(update: Update, context: ContextTypes.DEFAULT_TYPE) -> int:
    """دریافت گزینه صحیح."""
    query = update.callback_query
    await query.answer()
    
    if query.data == "admin_cancel":
        return await cancel_add_question_callback(update, context)
    
    # استخراج گزینه (correct_A -> A)
    correct = query.data.split("_")[1]
    context.user_data["new_question"]["correct_answer"] = correct
    
    await query.message.edit_text(
        "📝 *مرحله ۷ از ۷ - پاسخنامه تشریحی*\n\n"
        "توضیح تشریحی پاسخ را وارد کنید:\n"
        "(می‌توانید با «⏭ رد کردن» این مرحله را رد کنید)",
        parse_mode="Markdown"
    )
    
    await query.message.reply_text(
        "پاسخنامه تشریحی:",
        reply_markup=get_skip_keyboard()
    )
    return EXPLANATION


async def receive_explanation(update: Update, context: ContextTypes.DEFAULT_TYPE) -> int:
    """دریافت پاسخنامه تشریحی و ذخیره سوال."""
    text = update.message.text.strip()
    
    if text == "❌ انصراف":
        return await cancel_add_question(update, context)
    
    explanation = None if text == "⏭ رد کردن" else text
    
    q = context.user_data["new_question"]
    
    # ذخیره در دیتابیس
    question_id = await QuestionRepository.insert(
        text=q["text"],
        option_a=q["option_a"],
        option_b=q["option_b"],
        option_c=q["option_c"],
        option_d=q["option_d"],
        correct_answer=q["correct_answer"],
        explanation=explanation,
        created_by=update.effective_user.id
    )
    
    # ذخیره شناسه سوال برای لینک به پیش‌آزمون
    context.user_data["last_question_id"] = question_id
    
    # دریافت لیست پیش‌آزمون‌ها
    preexams = await PreexamRepository.get_all_active()
    
    if not preexams:
        # پاک کردن داده موقت
        context.user_data.pop("new_question", None)
        
        correct_labels = {"A": "الف", "B": "ب", "C": "ج", "D": "د"}
        
        # دکمه‌های عملیات سریع
        quick_buttons = InlineKeyboardMarkup([
            [InlineKeyboardButton("➕ افزودن سوال دیگر", callback_data="admin_add_question")],
            [InlineKeyboardButton("🔙 بازگشت به پنل", callback_data="back_to_admin_menu")],
        ])
        
        await update.message.reply_text(
            f"✅ *سوال ذخیره شد!*\n\n"
            f"🆔 شناسه: {question_id}\n"
            f"✔️ گزینه صحیح: {correct_labels[q['correct_answer']]}\n\n"
            f"⚠️ هیچ پیش‌آزمون فعالی وجود ندارد.",
            parse_mode="Markdown",
            reply_markup=ReplyKeyboardRemove()
        )
        await update.message.reply_text(
            "چه کاری انجام دهم؟",
            reply_markup=quick_buttons
        )
        return ConversationHandler.END

    
    # ساخت کیبورد انتخاب پیش‌آزمون
    keyboard = []
    for exam in preexams:
        keyboard.append([InlineKeyboardButton(
            f"📋 {exam['title']}",
            callback_data=f"link_preexam_{exam['id']}"
        )])
    keyboard.append([InlineKeyboardButton("⏭ رد کردن (بدون پیش‌آزمون)", callback_data="link_preexam_skip")])
    
    correct_labels = {"A": "الف", "B": "ب", "C": "ج", "D": "د"}
    
    await update.message.reply_text(
        f"✅ *سوال ذخیره شد!* (شناسه: {question_id})\n\n"
        f"✔️ گزینه صحیح: {correct_labels[q['correct_answer']]}\n\n"
        f"📋 این سوال را به کدام پیش‌آزمون اضافه کنید؟",
        parse_mode="Markdown",
        reply_markup=InlineKeyboardMarkup(keyboard)
    )
    
    # پاک کردن داده موقت
    context.user_data.pop("new_question", None)
    
    return SELECT_PREEXAM


async def handle_preexam_selection(update: Update, context: ContextTypes.DEFAULT_TYPE) -> int:
    """هندل انتخاب پیش‌آزمون برای لینک سوال."""
    query = update.callback_query
    await query.answer()
    
    question_id = context.user_data.get("last_question_id")
    
    # دکمه‌های عملیات سریع
    quick_buttons = InlineKeyboardMarkup([
        [InlineKeyboardButton("➕ افزودن سوال دیگر", callback_data="admin_add_question")],
        [InlineKeyboardButton("🔙 بازگشت به پنل", callback_data="back_to_admin_menu")],
    ])
    
    if query.data == "link_preexam_skip":
        await query.message.edit_text(
            f"✅ سوال ذخیره شد (بدون لینک به پیش‌آزمون).",
            reply_markup=quick_buttons
        )
        return ConversationHandler.END
    
    preexam_id = int(query.data.split("_")[2])
    
    # اضافه کردن سوال به پیش‌آزمون
    await PreexamRepository.add_question(preexam_id, question_id)
    
    # دریافت اطلاعات پیش‌آزمون
    preexam = await PreexamRepository.get_by_id(preexam_id)
    preexam_title = preexam['title'] if preexam else "نامشخص"
    
    await query.message.edit_text(
        f"✅ سوال با موفقیت به پیش‌آزمون «{preexam_title}» اضافه شد!",
        reply_markup=quick_buttons
    )
    
    return ConversationHandler.END



async def cancel_add_question(update: Update, context: ContextTypes.DEFAULT_TYPE) -> int:
    """انصراف از افزودن سوال."""
    context.user_data.pop("new_question", None)
    
    await update.message.reply_text(
        "❌ افزودن سوال لغو شد.\n\n"
        "برای بازگشت به پنل ادمین از /admin استفاده کنید.",
        reply_markup=ReplyKeyboardRemove()
    )
    return ConversationHandler.END


async def cancel_add_question_callback(update: Update, context: ContextTypes.DEFAULT_TYPE) -> int:
    """انصراف از افزودن سوال (از طریق کالبک)."""
    query = update.callback_query
    await query.answer()
    
    context.user_data.pop("new_question", None)
    
    await query.message.edit_text(
        "❌ افزودن سوال لغو شد.\n\n"
        "برای بازگشت به پنل ادمین از /admin استفاده کنید."
    )
    return ConversationHandler.END


@admin_only
async def list_questions_callback(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """نمایش لیست سوالات."""
    query = update.callback_query
    await query.answer()
    
    questions = await QuestionRepository.get_all_active()
    
    if not questions:
        await query.message.edit_text(
            "📋 هیچ سوالی در مخزن وجود ندارد.\n\n"
            "برای افزودن سوال از پنل ادمین استفاده کنید.",
            reply_markup=get_admin_menu_keyboard()
        )
        return
    
    # نمایش ۵ سوال آخر
    text = "📋 *لیست آخرین سوالات:*\n\n"
    for q in questions[-5:]:
        correct_labels = {"A": "الف", "B": "ب", "C": "ج", "D": "د"}
        text += f"🆔 *{q.id}*: {q.text[:50]}{'...' if len(q.text) > 50 else ''}\n"
        text += f"   ✔️ پاسخ: {correct_labels[q.correct_answer]}\n\n"
    
    text += f"📊 مجموع: {len(questions)} سوال"
    
    await query.message.edit_text(
        text,
        parse_mode="Markdown",
        reply_markup=get_admin_menu_keyboard()
    )


async def admin_back_callback(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """بازگشت از پنل ادمین."""
    query = update.callback_query
    await query.answer()
    await query.message.edit_text("🔙 بازگشت به منوی اصلی.\n\nبرای بازگشت به پنل ادمین /admin را ارسال کنید.")


# ==================== هندلرهای بررسی سوالات ====================

@admin_only
async def show_pending_questions(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """نمایش سوالات در انتظار تایید."""
    query = update.callback_query
    await query.answer()
    
    pending = await PendingQuestionRepository.get_pending()
    
    if not pending:
        await query.message.edit_text(
            "📥 هیچ سوالی در انتظار تایید نیست.",
            reply_markup=get_admin_menu_keyboard()
        )
        return
    
    # ذخیره لیست سوالات در context
    context.user_data["pending_questions"] = pending
    context.user_data["pending_index"] = 0
    
    # نمایش اولین سوال
    await show_pending_question(query, context, 0)


async def show_pending_question(query, context: ContextTypes.DEFAULT_TYPE, index: int) -> None:
    """نمایش یک سوال در انتظار."""
    pending = context.user_data.get("pending_questions", [])
    
    if not pending or index >= len(pending):
        return
    
    q = pending[index]
    option_labels = {"A": "الف", "B": "ب", "C": "ج", "D": "د"}
    
    text = f"""
📥 *سوال {index + 1} از {len(pending)} - در انتظار تایید*

🆔 شناسه: {q['id']}
👤 ارسال‌کننده: {q.get('submitter_name', 'ناشناس')}

📝 *سوال:*
{q['text']}

*الف)* {q['option_a']}
*ب)* {q['option_b']}
*ج)* {q['option_c']}
*د)* {q['option_d']}

✔️ *پاسخ صحیح:* {option_labels[q['correct_answer']]}
"""
    
    if q.get('explanation'):
        text += f"\n📖 *توضیح:*\n{q['explanation']}"
    
    keyboard = [
        [
            InlineKeyboardButton("✅ تایید", callback_data=f"approve_{q['id']}"),
            InlineKeyboardButton("❌ رد", callback_data=f"reject_{q['id']}"),
        ]
    ]
    
    # دکمه‌های ناویگیشن
    nav_row = []
    if index > 0:
        nav_row.append(InlineKeyboardButton("⬅️ قبلی", callback_data="pending_prev"))
    if index < len(pending) - 1:
        nav_row.append(InlineKeyboardButton("➡️ بعدی", callback_data="pending_next"))
    if nav_row:
        keyboard.append(nav_row)
    
    keyboard.append([InlineKeyboardButton("🔙 بازگشت به پنل", callback_data="back_to_admin_menu")])
    
    await query.message.edit_text(
        text,
        parse_mode="Markdown",
        reply_markup=InlineKeyboardMarkup(keyboard)
    )


@admin_only
async def approve_question(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """تایید سوال."""
    query = update.callback_query
    await query.answer()
    
    question_id = int(query.data.split("_")[1])
    admin_id = update.effective_user.id
    
    success = await PendingQuestionRepository.approve(question_id, admin_id)
    
    if success:
        await query.message.edit_text(
            f"✅ سوال {question_id} تایید شد و به مخزن تست شانسی اضافه شد.",
            reply_markup=InlineKeyboardMarkup([[
                InlineKeyboardButton("📥 سوال بعدی", callback_data="admin_pending"),
                InlineKeyboardButton("🔙 پنل ادمین", callback_data="back_to_admin_menu")
            ]])
        )
    else:
        await query.answer("❌ خطا در تایید سوال", show_alert=True)


@admin_only
async def reject_question(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """رد سوال."""
    query = update.callback_query
    await query.answer()
    
    question_id = int(query.data.split("_")[1])
    admin_id = update.effective_user.id
    
    success = await PendingQuestionRepository.reject(question_id, admin_id)
    
    if success:
        await query.message.edit_text(
            f"❌ سوال {question_id} رد شد.",
            reply_markup=InlineKeyboardMarkup([[
                InlineKeyboardButton("📥 سوال بعدی", callback_data="admin_pending"),
                InlineKeyboardButton("🔙 پنل ادمین", callback_data="back_to_admin_menu")
            ]])
        )
    else:
        await query.answer("❌ خطا در رد سوال", show_alert=True)


async def pending_nav(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """ناویگیشن در سوالات در انتظار."""
    query = update.callback_query
    await query.answer()
    
    pending = context.user_data.get("pending_questions", [])
    current_index = context.user_data.get("pending_index", 0)
    
    if query.data == "pending_next":
        current_index = min(current_index + 1, len(pending) - 1)
    elif query.data == "pending_prev":
        current_index = max(current_index - 1, 0)
    
    context.user_data["pending_index"] = current_index
    await show_pending_question(query, context, current_index)


async def back_to_admin_menu(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """بازگشت به منوی ادمین."""
    query = update.callback_query
    await query.answer()
    
    question_count = await QuestionRepository.get_count()
    pending_count = await PendingQuestionRepository.get_pending_count()
    user_question_count = await UserQuestionRepository.get_count()
    
    await query.message.edit_text(
        f"🔐 *پنل مدیریت*\n\n"
        f"📝 سوالات مخزن A (آزمون): {question_count}\n"
        f"📥 سوالات در انتظار تایید: {pending_count}\n"
        f"🎲 سوالات مخزن B (شانسی): {user_question_count}\n\n"
        f"از منوی زیر انتخاب کنید:",
        parse_mode="Markdown",
        reply_markup=get_admin_menu_keyboard()
    )


# ==================== هندلرهای مدیریت ثبت‌نام‌ها ====================

@admin_only
async def show_registrants_list(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """نمایش لیست ثبت‌نام‌کنندگان در انتظار."""
    query = update.callback_query
    await query.answer()
    
    pending = await ServiceRequestRepository.get_pending()
    
    if not pending:
        # حذف پیام قبلی (ممکن است عکس باشد)
        try:
            await query.message.delete()
        except:
            pass
        await context.bot.send_message(
            chat_id=update.effective_chat.id,
            text="👥 هیچ درخواست ثبت‌نامی در انتظار بررسی نیست.",
            reply_markup=get_admin_menu_keyboard()
        )
        return
    
    keyboard = []
    for req in pending:
        user_name = req.get('full_name', 'ناشناس')
        service = req.get('service_type', 'نامشخص')
        keyboard.append([InlineKeyboardButton(
            f"👤 {user_name} - {service[:15]}...",
            callback_data=f"admin_reg_{req['id']}"
        )])
    
    keyboard.append([InlineKeyboardButton("🔙 بازگشت", callback_data="admin_back")])
    
    # حذف پیام قبلی و ارسال پیام جدید (چون ممکن است عکس باشد)
    try:
        await query.message.delete()
    except:
        pass
    
    await context.bot.send_message(
        chat_id=update.effective_chat.id,
        text=f"👥 *لیست درخواست‌های در انتظار*\n\n"
             f"تعداد: {len(pending)}\n\n"
             "برای بررسی جزئیات روی نام کاربر کلیک کنید:",
        parse_mode="Markdown",
        reply_markup=InlineKeyboardMarkup(keyboard)
    )



@admin_only
async def show_registrant_detail(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """نمایش جزئیات درخواست ثبت‌نام."""
    query = update.callback_query
    await query.answer()
    
    request_id = int(query.data.split("_")[2])
    req = await ServiceRequestRepository.get_by_id(request_id)
    
    if not req:
        await query.answer("❌ درخواست یافت نشد", show_alert=True)
        return
    
    context.user_data["current_reg_id"] = request_id
    context.user_data["current_reg_user_id"] = req["user_id"]
    
    # دریافت اطلاعات پروفایل کاربر
    user = await UserRepository.get_by_telegram_id(req["user_id"])
    
    # شماره تلگرام (اگر ثبت شده باشد)
    telegram_phone = user.phone_number if user and user.phone_number else "ثبت نشده"
    
    # دریافت آیدی تلگرام از API
    try:
        tg_user = await context.bot.get_chat(req["user_id"])
        if tg_user.username:
            escaped_username = tg_user.username.replace("_", "\\_")
            telegram_username = f"@{escaped_username}"
        else:
            telegram_username = "آیدی ندارد"
    except:
        telegram_username = "آیدی ندارد"
    
    # متن جزئیات
    text = f"""
👤 *جزئیات درخواست ثبت‌نام*

🆔 شناسه درخواست: {req['id']}

━━━━━━ *اطلاعات کاربر* ━━━━━━
👤 نام: {req.get('full_name', '---')}
🆔 آیدی تلگرام: {telegram_username}
📍 استان: {req.get('province', '---')}
🎓 پایه: {req.get('grade', '---')}
📚 رشته: {req.get('field', '---')}

━━━━━━ *شماره تماس* ━━━━━━
📱 شماره تلگرام: {telegram_phone}
📞 شماره وارد شده: {req.get('phone', '---')}

━━━━━━ *درخواست* ━━━━━━
📋 سرویس: {req.get('service_type', '---')}
📅 تاریخ: {req.get('created_at', '---')}

📊 وضعیت: {req.get('status', '---')}
"""
    
    keyboard = [
        [
            InlineKeyboardButton("✅ تایید", callback_data=f"reg_approve_{request_id}"),
            InlineKeyboardButton("❌ رد", callback_data=f"reg_reject_{request_id}"),
        ],
        [InlineKeyboardButton("✉️ ارسال پیام به کاربر", callback_data=f"reg_msg_{request_id}")],
        [InlineKeyboardButton("🔙 بازگشت به لیست", callback_data="admin_registrants")],
    ]
    
    # اگر عکس دارد ارسال کن، اگر نه پیام متنی
    if req.get('receipt_file_id'):
        await query.message.delete() # حذف پیام قبلی برای ارسال عکس جدید
        await context.bot.send_photo(
            chat_id=update.effective_chat.id,
            photo=req['receipt_file_id'],
            caption=text,
            parse_mode="Markdown",
            reply_markup=InlineKeyboardMarkup(keyboard)
        )
    else:
        await query.message.edit_text(
            text,
            parse_mode="Markdown",
            reply_markup=InlineKeyboardMarkup(keyboard)
        )



@admin_only
async def approve_registrant(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """تایید درخواست ثبت‌نام."""
    query = update.callback_query
    await query.answer()
    
    request_id = int(query.data.split("_")[2])
    req = await ServiceRequestRepository.get_by_id(request_id)
    
    if await ServiceRequestRepository.update_status(request_id, "approved", 1):
        # اطلاع به کاربر
        try:
            await context.bot.send_message(
                chat_id=req['user_id'],
                text=f"✅ رسید واریزی شما برای «{req['service_type']}» تایید شد.\n\n"
                     f"📍 واحد پشتیبانی تراز در اسرع وقت با شما تماس خواهد گرفت. از صبر و همکاری شما سپاسگزاریم.",
                parse_mode="Markdown"
            )
        except Exception:
            pass

        
        await query.message.delete()
        await context.bot.send_message(
            chat_id=update.effective_chat.id,
            text=f"✅ درخواست {request_id} تایید شد.",
            reply_markup=InlineKeyboardMarkup([[InlineKeyboardButton("🔙 بازگشت به لیست", callback_data="admin_registrants")]])
        )


@admin_only
async def reject_registrant(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """رد درخواست ثبت‌نام."""
    query = update.callback_query
    await query.answer()
    
    request_id = int(query.data.split("_")[2])
    req = await ServiceRequestRepository.get_by_id(request_id)
    
    if await ServiceRequestRepository.update_status(request_id, "rejected", 0):
        # اطلاع به کاربر
        try:
            await context.bot.send_message(
                chat_id=req['user_id'],
                text=f"❌ متاسفانه درخواست ثبت‌نام شما برای «{req['service_type']}» رد شد.\n"
                     f"جهت پیگیری می‌توانید با پشتیبانی تماس بگیرید.",
                parse_mode="Markdown"
            )
        except Exception:
            pass

        await query.message.delete()
        await context.bot.send_message(
            chat_id=update.effective_chat.id,
            text=f"❌ درخواست {request_id} رد شد.",
            reply_markup=InlineKeyboardMarkup([[InlineKeyboardButton("🔙 بازگشت به لیست", callback_data="admin_registrants")]])
        )


@admin_only
async def start_message_registrant(update: Update, context: ContextTypes.DEFAULT_TYPE) -> int:
    """شروع ارسال پیام به کاربر."""
    query = update.callback_query
    await query.answer()
    
    request_id = int(query.data.split("_")[2])
    context.user_data["msg_req_id"] = request_id
    
    # نیاز داریم user_id را پیدا کنیم
    req = await ServiceRequestRepository.get_by_id(request_id)
    if not req:
        await query.answer("خطا در یافتن درخواست")
        return ConversationHandler.END
        
    context.user_data["msg_user_id"] = req["user_id"]
    
    await context.bot.send_message(
        chat_id=update.effective_chat.id,
        text="✉️ لطفاً متن پیام خود را برای این کاربر ارسال کنید:",
        reply_markup=get_cancel_keyboard()
    )
    return REPLY_TEXT


async def send_message_registrant(update: Update, context: ContextTypes.DEFAULT_TYPE) -> int:
    """ارسال پیام نهایی به کاربر."""
    text = update.message.text
    
    if text == "❌ انصراف":
        await update.message.reply_text("ارسال پیام لغو شد.", reply_markup=ReplyKeyboardRemove())
        return ConversationHandler.END
        
    user_id = context.user_data.get("msg_user_id")
    
    try:
        await context.bot.send_message(
            chat_id=user_id,
            text=f"📩 *پیام از طرف مدیریت:*\n\n{text}",
            parse_mode="Markdown"
        )
        await update.message.reply_text("✅ پیام با موفقیت ارسال شد.", reply_markup=ReplyKeyboardRemove())
    except Exception as e:
        await update.message.reply_text(f"❌ خطا در ارسال پیام: {e}", reply_markup=ReplyKeyboardRemove())
        
    return ConversationHandler.END


# ==================== هندلرهای رسیدهای تایید شده ====================

@admin_only
async def show_approved_list(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """نمایش لیست رسیدهای تایید شده."""
    query = update.callback_query
    await query.answer()
    
    approved = await ServiceRequestRepository.get_approved()
    
    if not approved:
        await query.message.edit_text(
            "✅ هیچ رسید تایید شده‌ای وجود ندارد.",
            reply_markup=InlineKeyboardMarkup([
                [InlineKeyboardButton("🔙 بازگشت", callback_data="admin_products_mgmt")]
            ])
        )
        return
    
    keyboard = []
    for req in approved[:15]:
        service_short = req['service_type'][:20] + "..." if len(req['service_type']) > 20 else req['service_type']
        keyboard.append([InlineKeyboardButton(
            f"#{req['id']} - {req.get('full_name', 'ناشناس')} - {service_short}",
            callback_data=f"admin_approved_{req['id']}"
        )])
    
    keyboard.append([InlineKeyboardButton("🔙 بازگشت", callback_data="admin_products_mgmt")])
    
    await query.message.edit_text(
        f"✅ *رسیدهای تایید شده ({len(approved)})*\n\n"
        f"برای مشاهده جزییات روی هر مورد کلیک کنید:",
        parse_mode="Markdown",
        reply_markup=InlineKeyboardMarkup(keyboard)
    )


@admin_only
async def show_approved_detail(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """نمایش جزییات رسید تایید شده."""
    query = update.callback_query
    await query.answer()
    
    request_id = int(query.data.split("_")[2])
    req = await ServiceRequestRepository.get_by_id(request_id)
    
    if not req:
        await query.answer("درخواست یافت نشد", show_alert=True)
        return
    
    text = (
        f"✅ *رسید تایید شده*\n\n"
        f"🆔 شناسه: {req['id']}\n"
        f"👤 نام: {req.get('full_name', 'ناشناس')}\n"
        f"📞 شماره: {req['phone']}\n"
        f"📋 سرویس: {req['service_type']}\n"
        f"📅 تاریخ: {req.get('created_at', '---')}"
    )
    
    keyboard = [
        [InlineKeyboardButton("🗑 حذف", callback_data=f"del_approved_{request_id}")],
        [InlineKeyboardButton("🔙 بازگشت به لیست", callback_data="admin_approved")],
    ]
    
    if req.get('receipt_file_id'):
        await query.message.delete()
        await context.bot.send_photo(
            chat_id=update.effective_chat.id,
            photo=req['receipt_file_id'],
            caption=text,
            parse_mode="Markdown",
            reply_markup=InlineKeyboardMarkup(keyboard)
        )
    else:
        await query.message.edit_text(
            text,
            parse_mode="Markdown",
            reply_markup=InlineKeyboardMarkup(keyboard)
        )


@admin_only
async def delete_approved(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """حذف رسید تایید شده."""
    query = update.callback_query
    await query.answer()
    
    request_id = int(query.data.split("_")[2])
    
    if await ServiceRequestRepository.delete(request_id):
        await query.message.delete()
        await context.bot.send_message(
            chat_id=update.effective_chat.id,
            text=f"🗑 رسید شماره {request_id} با موفقیت حذف شد.",
            reply_markup=InlineKeyboardMarkup([
                [InlineKeyboardButton("🔙 بازگشت به لیست", callback_data="admin_approved")]
            ])
        )


# ==================== هندلرهای ارسال پست به کانال ====================

@admin_only
async def start_channel_post(update: Update, context: ContextTypes.DEFAULT_TYPE) -> int:
    """شروع فرآیند ارسال پست به کانال."""
    query = update.callback_query
    await query.answer()
    
    await query.message.edit_text(
        "📢 *ارسال پست به کانال*\n\n"
        "متن پیامی که می‌خواهید در کانال ارسال شود را بنویسید:\n\n"
        "💡 می‌توانید از فرمت Markdown استفاده کنید.",
        parse_mode="Markdown"
    )
    await context.bot.send_message(
        chat_id=update.effective_chat.id,
        text="متن پیام:",
        reply_markup=get_cancel_keyboard()
    )
    return POST_MESSAGE_TEXT


async def receive_post_message(update: Update, context: ContextTypes.DEFAULT_TYPE) -> int:
    """دریافت متن پیام کانال."""
    text = update.message.text.strip()
    
    if text == "❌ انصراف":
        await update.message.reply_text(
            "❌ ارسال پست لغو شد.",
            reply_markup=ReplyKeyboardRemove()
        )
        return ConversationHandler.END
    
    context.user_data["channel_post_message"] = text
    
    await update.message.reply_text(
        "✅ متن ذخیره شد.\n\n"
        "حالا متن دکمه شیشه‌ای را بنویسید:\n"
        "(مثال: 🚀 شرکت در پیش‌آزمون)",
        reply_markup=get_cancel_keyboard()
    )
    return POST_BUTTON_TEXT


async def receive_button_text(update: Update, context: ContextTypes.DEFAULT_TYPE) -> int:
    """دریافت متن دکمه."""
    text = update.message.text.strip()
    
    if text == "❌ انصراف":
        await update.message.reply_text(
            "❌ ارسال پست لغو شد.",
            reply_markup=ReplyKeyboardRemove()
        )
        return ConversationHandler.END
    
    context.user_data["channel_post_button"] = text
    
    # دریافت لیست پیش‌آزمون‌ها
    preexams = await PreexamRepository.get_all_active()
    
    if not preexams:
        await update.message.reply_text(
            "❌ هیچ پیش‌آزمون فعالی وجود ندارد.\n"
            "ابتدا از بخش مدیریت پیش‌آزمون‌ها یک پیش‌آزمون ایجاد کنید.",
            reply_markup=ReplyKeyboardRemove()
        )
        return ConversationHandler.END
    
    keyboard = []
    for exam in preexams:
        keyboard.append([InlineKeyboardButton(
            f"📋 {exam['title']}",
            callback_data=f"post_preexam_{exam['id']}"
        )])
    keyboard.append([InlineKeyboardButton("❌ انصراف", callback_data="post_cancel")])
    
    # حذف کیبورد ReplyKeyboard قبل از نمایش InlineKeyboard
    await update.message.reply_text(
        "📋 در حال بارگذاری پیش‌آزمون‌ها...",
        reply_markup=ReplyKeyboardRemove()
    )
    
    await context.bot.send_message(
        chat_id=update.effective_chat.id,
        text="🎯 کدام پیش‌آزمون با این دکمه لینک شود؟",
        reply_markup=InlineKeyboardMarkup(keyboard)
    )
    
    return POST_SELECT_PREEXAM



async def select_preexam_for_post(update: Update, context: ContextTypes.DEFAULT_TYPE) -> int:
    """انتخاب پیش‌آزمون و ارسال به کانال."""
    query = update.callback_query
    await query.answer()
    
    if query.data == "post_cancel":
        await query.message.edit_text("❌ ارسال پست لغو شد.")
        return ConversationHandler.END
    
    preexam_id = int(query.data.split("_")[2])
    preexam = await PreexamRepository.get_by_id(preexam_id)
    
    message_text = context.user_data.get("channel_post_message", "")
    button_text = context.user_data.get("channel_post_button", "شرکت در پیش‌آزمون")
    
    # ایجاد دیپ لینک
    bot_info = await context.bot.get_me()
    deep_link = f"https://t.me/{bot_info.username}?start=preexam_{preexam_id}"
    
    keyboard = InlineKeyboardMarkup([
        [InlineKeyboardButton(button_text, url=deep_link)]
    ])
    
    # ارسال به کانال
    try:
        from config import REQUIRED_CHANNEL_ID
        await context.bot.send_message(
            chat_id=REQUIRED_CHANNEL_ID,
            text=message_text,
            parse_mode="Markdown",
            reply_markup=keyboard
        )
        
        await query.message.edit_text(
            f"✅ پست با موفقیت به کانال ارسال شد!\n\n"
            f"📋 پیش‌آزمون: {preexam['title']}\n"
            f"🔗 لینک: {deep_link}"
        )
    except Exception as e:
        await query.message.edit_text(f"❌ خطا در ارسال به کانال:\n{e}")
    
    # پاک کردن داده‌های موقت
    context.user_data.pop("channel_post_message", None)
    context.user_data.pop("channel_post_button", None)
    
    return ConversationHandler.END


# ==================== هندلرهای پیام همگانی ====================

@admin_only
async def start_broadcast(update: Update, context: ContextTypes.DEFAULT_TYPE) -> int:
    """شروع فرآیند ارسال پیام همگانی."""
    query = update.callback_query
    await query.answer()
    
    # شمارش تعداد کاربران
    all_users = await UserRepository.get_all()
    user_count = len(all_users)
    
    await query.message.edit_text(
        f"📢 *ارسال پیام همگانی*\n\n"
        f"👥 تعداد کاربران: {user_count}\n\n"
        f"پیام خود را ارسال کنید. می‌تواند شامل موارد زیر باشد:\n"
        f"• متن\n"
        f"• عکس\n"
        f"• ویدیو\n"
        f"• فایل\n"
        f"• استیکر\n\n"
        f"⚠️ پیام شما به همه کاربران فوروارد می‌شود.",
        parse_mode="Markdown"
    )
    await context.bot.send_message(
        chat_id=update.effective_chat.id,
        text="پیام همگانی:",
        reply_markup=get_cancel_keyboard()
    )
    return BROADCAST_MESSAGE


async def receive_broadcast_message(update: Update, context: ContextTypes.DEFAULT_TYPE) -> int:
    """دریافت پیام همگانی و نمایش پیش‌نمایش."""
    
    # چک انصراف
    if update.message.text and update.message.text.strip() == "❌ انصراف":
        await update.message.reply_text(
            "❌ ارسال پیام همگانی لغو شد.",
            reply_markup=ReplyKeyboardRemove()
        )
        return ConversationHandler.END
    
    # ذخیره اطلاعات پیام برای ارسال بعدی
    context.user_data["broadcast_chat_id"] = update.effective_chat.id
    context.user_data["broadcast_message_id"] = update.message.message_id
    
    # دریافت تعداد کاربران
    all_users = await UserRepository.get_all()
    user_count = len(all_users)
    
    # نمایش پیش‌نمایش
    await update.message.reply_text(
        "👆 *پیش‌نمایش پیام بالا*\n\n"
        f"👥 تعداد دریافت‌کنندگان: {user_count}\n\n"
        "آیا از ارسال این پیام به همه کاربران اطمینان دارید؟",
        parse_mode="Markdown",
        reply_markup=InlineKeyboardMarkup([
            [InlineKeyboardButton("✅ تایید و ارسال", callback_data="broadcast_confirm")],
            [InlineKeyboardButton("❌ انصراف", callback_data="broadcast_cancel")],
        ])
    )
    
    return BROADCAST_CONFIRM


async def confirm_broadcast(update: Update, context: ContextTypes.DEFAULT_TYPE) -> int:
    """تایید و ارسال پیام همگانی."""
    query = update.callback_query
    await query.answer()
    
    if query.data == "broadcast_cancel":
        await query.message.edit_text("❌ ارسال پیام همگانی لغو شد.")
        return ConversationHandler.END
    
    # دریافت اطلاعات پیام ذخیره شده
    broadcast_chat_id = context.user_data.get("broadcast_chat_id")
    broadcast_message_id = context.user_data.get("broadcast_message_id")
    
    if not broadcast_chat_id or not broadcast_message_id:
        await query.message.edit_text("❌ خطا: پیام یافت نشد.")
        return ConversationHandler.END
    
    # دریافت همه کاربران
    all_users = await UserRepository.get_all()
    
    success_count = 0
    fail_count = 0
    
    await query.message.edit_text(
        f"📤 در حال ارسال...\n\n"
        f"✅ موفق: 0\n"
        f"❌ ناموفق: 0"
    )
    
    for user in all_users:
        try:
            # کپی کردن پیام بدون نمایش فرستنده اصلی
            await context.bot.copy_message(
                chat_id=user.telegram_id,
                from_chat_id=broadcast_chat_id,
                message_id=broadcast_message_id
            )
            success_count += 1
        except Exception:
            fail_count += 1
        
        # آپدیت وضعیت هر 10 پیام
        if (success_count + fail_count) % 10 == 0:
            try:
                await query.message.edit_text(
                    f"📤 در حال ارسال...\n\n"
                    f"✅ موفق: {success_count}\n"
                    f"❌ ناموفق: {fail_count}"
                )
            except:
                pass
    
    # پیام نهایی
    await query.message.edit_text(
        f"✅ *ارسال پیام همگانی تکمیل شد*\n\n"
        f"📊 آمار ارسال:\n"
        f"✅ موفق: {success_count}\n"
        f"❌ ناموفق: {fail_count}\n"
        f"📨 کل: {success_count + fail_count}",
        parse_mode="Markdown"
    )
    
    # پاک کردن داده‌های موقت
    context.user_data.pop("broadcast_chat_id", None)
    context.user_data.pop("broadcast_message_id", None)
    
    return ConversationHandler.END


# ==================== هندلرهای معرفی مجموعه ====================

@admin_only
async def start_set_about(update: Update, context: ContextTypes.DEFAULT_TYPE) -> int:
    """شروع تنظیم معرفی مجموعه."""
    query = update.callback_query
    await query.answer()
    
    await query.message.edit_text(
        "ℹ️ *تنظیم معرفی مجموعه*\n\n"
        "پیامی که می‌خواهید به عنوان معرفی مجموعه نمایش داده شود را ارسال کنید.\n\n"
        "می‌تواند شامل موارد زیر باشد:\n"
        "• متن\n"
        "• عکس با کپشن\n"
        "• ویدیو\n\n"
        "⚠️ این پیام برای کاربرانی که روی «معرفی مجموعه» کلیک می‌کنند فوروارد می‌شود.",
        parse_mode="Markdown"
    )
    await context.bot.send_message(
        chat_id=update.effective_chat.id,
        text="پیام معرفی مجموعه:",
        reply_markup=get_cancel_keyboard()
    )
    return ABOUT_CONTENT


async def receive_about_content(update: Update, context: ContextTypes.DEFAULT_TYPE) -> int:
    """دریافت و ذخیره محتوای معرفی مجموعه."""
    
    # چک انصراف
    if update.message.text and update.message.text.strip() == "❌ انصراف":
        await update.message.reply_text(
            "❌ تنظیم معرفی مجموعه لغو شد.",
            reply_markup=ReplyKeyboardRemove()
        )
        return ConversationHandler.END
    
    # ذخیره اطلاعات پیام
    admin_chat_id = update.effective_chat.id
    message_id = update.message.message_id
    
    # ذخیره در تنظیمات
    await SettingsRepository.set("about_chat_id", str(admin_chat_id))
    await SettingsRepository.set("about_message_id", str(message_id))
    
    await update.message.reply_text(
        "✅ *معرفی مجموعه با موفقیت ذخیره شد!*\n\n"
        "اکنون کاربرانی که روی دکمه «ℹ️ معرفی مجموعه» کلیک کنند، این پیام را دریافت می‌کنند.",
        parse_mode="Markdown",
        reply_markup=ReplyKeyboardRemove()
    )
    
    return ConversationHandler.END


# ==================== تابع دریافت هندلرها ====================


def get_admin_handlers() -> list:
    """برگرداندن لیست هندلرهای ادمین."""
    # ConversationHandler برای افزودن سوال
    add_question_handler = ConversationHandler(
        entry_points=[
            CallbackQueryHandler(start_add_question, pattern="^admin_add_question$"),
        ],
        states={
            QUESTION_TEXT: [
                MessageHandler(filters.TEXT & ~filters.COMMAND, receive_question_text),
            ],
            OPTION_A: [
                MessageHandler(filters.TEXT & ~filters.COMMAND, receive_option_a),
            ],
            OPTION_B: [
                MessageHandler(filters.TEXT & ~filters.COMMAND, receive_option_b),
            ],
            OPTION_C: [
                MessageHandler(filters.TEXT & ~filters.COMMAND, receive_option_c),
            ],
            OPTION_D: [
                MessageHandler(filters.TEXT & ~filters.COMMAND, receive_option_d),
            ],
            CORRECT_ANSWER: [
                CallbackQueryHandler(receive_correct_answer, pattern="^(correct_[ABCD]|admin_cancel)$"),
            ],
            EXPLANATION: [
                MessageHandler(filters.TEXT & ~filters.COMMAND, receive_explanation),
            ],
            SELECT_PREEXAM: [
                CallbackQueryHandler(handle_preexam_selection, pattern=r"^link_preexam_"),
            ],
        },
        fallbacks=[
            MessageHandler(filters.Regex("^❌ انصراف$"), cancel_add_question),
            CallbackQueryHandler(cancel_add_question_callback, pattern="^admin_cancel$"),
        ],
        allow_reentry=True,
    )

    

    # ConversationHandler برای ارسال پیام
    msg_registrant_handler = ConversationHandler(
        entry_points=[
            CallbackQueryHandler(start_message_registrant, pattern=r"^reg_msg_\d+$"),
        ],
        states={
            REPLY_TEXT: [
                MessageHandler(filters.TEXT & ~filters.COMMAND, send_message_registrant),
            ],
        },
        fallbacks=[
            MessageHandler(filters.Regex("^❌ انصراف$"), cancel_add_question),
        ],
        allow_reentry=True,
    )

    # ConversationHandler برای تنظیم قیمت
    price_setting_handler = ConversationHandler(
        entry_points=[
            CallbackQueryHandler(start_set_exam_price, pattern="^admin_set_exam_price$"),
            CallbackQueryHandler(start_set_consult_price, pattern="^admin_set_consult_price$"),
        ],
        states={
            REPLY_TEXT: [
                MessageHandler(filters.TEXT & ~filters.COMMAND, receive_new_price),
            ],
        },
        fallbacks=[
            MessageHandler(filters.Regex("^❌ انصراف$"), cancel_add_question),
        ],
        allow_reentry=True,
    )

    # ConversationHandler برای ارسال پست به کانال
    channel_post_handler = ConversationHandler(
        entry_points=[
            CallbackQueryHandler(start_channel_post, pattern="^admin_channel_post$"),
        ],
        states={
            POST_MESSAGE_TEXT: [
                MessageHandler(filters.TEXT & ~filters.COMMAND, receive_post_message),
            ],
            POST_BUTTON_TEXT: [
                MessageHandler(filters.TEXT & ~filters.COMMAND, receive_button_text),
            ],
            POST_SELECT_PREEXAM: [
                CallbackQueryHandler(select_preexam_for_post, pattern=r"^post_"),
            ],
        },
        fallbacks=[
            MessageHandler(filters.Regex("^❌ انصراف$"), cancel_add_question),
        ],
        allow_reentry=True,
    )

    # ConversationHandler برای پیام همگانی
    broadcast_handler = ConversationHandler(
        entry_points=[
            CallbackQueryHandler(start_broadcast, pattern="^admin_broadcast$"),
        ],
        states={
            BROADCAST_MESSAGE: [
                MessageHandler(filters.ALL & ~filters.COMMAND, receive_broadcast_message),
            ],
            BROADCAST_CONFIRM: [
                CallbackQueryHandler(confirm_broadcast, pattern=r"^broadcast_"),
            ],
        },
        fallbacks=[
            MessageHandler(filters.Regex("^❌ انصراف$"), cancel_add_question),
        ],
        allow_reentry=True,
    )


    # ConversationHandler برای تنظیم معرفی مجموعه
    about_handler = ConversationHandler(
        entry_points=[
            CallbackQueryHandler(start_set_about, pattern="^admin_set_about$"),
        ],
        states={
            ABOUT_CONTENT: [
                MessageHandler(filters.ALL & ~filters.COMMAND, receive_about_content),
            ],
        },
        fallbacks=[
            MessageHandler(filters.Regex("^❌ انصراف$"), cancel_add_question),
        ],
        allow_reentry=True,
    )

    return [
        CommandHandler("admin", admin_command),
        add_question_handler,
        msg_registrant_handler,
        price_setting_handler,
        channel_post_handler,
        broadcast_handler,
        about_handler,

        CallbackQueryHandler(admin_stats_callback, pattern="^admin_stats$"),
        CallbackQueryHandler(list_questions_callback, pattern="^admin_list_questions$"),
        CallbackQueryHandler(admin_back_callback, pattern="^admin_back$"),
        # منوهای فرعی جدید
        CallbackQueryHandler(show_exam_management, pattern="^admin_exam_mgmt$"),
        CallbackQueryHandler(show_products_management, pattern="^admin_products_mgmt$"),
        CallbackQueryHandler(show_price_settings, pattern="^admin_price_settings$"),
        CallbackQueryHandler(show_pending_questions, pattern="^admin_pending$"),
        CallbackQueryHandler(approve_question, pattern=r"^approve_\d+$"),
        CallbackQueryHandler(reject_question, pattern=r"^reject_\d+$"),
        CallbackQueryHandler(pending_nav, pattern="^pending_(next|prev)$"),
        CallbackQueryHandler(back_to_admin_menu, pattern="^back_to_admin_menu$"),
        # هندلرهای مدیریت ثبت‌نام
        CallbackQueryHandler(show_registrants_list, pattern="^admin_registrants$"),
        CallbackQueryHandler(show_registrant_detail, pattern=r"^admin_reg_\d+$"),
        CallbackQueryHandler(approve_registrant, pattern=r"^reg_approve_\d+$"),
        CallbackQueryHandler(reject_registrant, pattern=r"^reg_reject_\d+$"),
        # هندلرهای رسیدهای تایید شده
        CallbackQueryHandler(show_approved_list, pattern="^admin_approved$"),
        CallbackQueryHandler(show_approved_detail, pattern=r"^admin_approved_\d+$"),
        CallbackQueryHandler(delete_approved, pattern=r"^del_approved_\d+$"),
        # هندلرهای مدیریت پیش‌آزمون
        CallbackQueryHandler(show_preexam_management, pattern="^admin_preexams$"),
        CallbackQueryHandler(create_preexam_start, pattern="^admin_create_preexam$"),
        CallbackQueryHandler(select_preexam_to_manage, pattern=r"^admin_preexam_\d+$"),
        CallbackQueryHandler(add_question_to_preexam_start, pattern=r"^preexam_add_q_\d+$"),
        CallbackQueryHandler(add_question_to_preexam, pattern=r"^preexam_addq_\d+_\d+$"),
        CallbackQueryHandler(view_preexam_questions, pattern=r"^preexam_view_q_\d+$"),
        CallbackQueryHandler(delete_preexam, pattern=r"^preexam_delete_\d+$"),
        # هندلرهای مخزن سوالات
        CallbackQueryHandler(show_questions_repo, pattern="^admin_questions_repo$"),
        CallbackQueryHandler(show_lucky_repo, pattern="^admin_lucky_repo$"),
        CallbackQueryHandler(delete_question_from_repo, pattern=r"^del_q_\d+$"),
        CallbackQueryHandler(delete_lucky_question, pattern=r"^del_lucky_\d+$"),
    ]


# ==================== هندلرهای مدیریت پیش‌آزمون ====================

@admin_only
async def show_preexam_management(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """نمایش مدیریت پیش‌آزمون‌ها."""
    query = update.callback_query
    await query.answer()
    
    preexams = await PreexamRepository.get_all_active()
    
    keyboard = [[InlineKeyboardButton("➕ ایجاد پیش‌آزمون جدید", callback_data="admin_create_preexam")]]
    
    for exam in preexams:
        q_count = await PreexamRepository.get_question_count(exam["id"])
        keyboard.append([InlineKeyboardButton(
            f"📋 {exam['title']} ({q_count} سوال)",
            callback_data=f"admin_preexam_{exam['id']}"
        )])
    
    keyboard.append([InlineKeyboardButton("🔙 بازگشت", callback_data="back_to_admin_menu")])
    
    await query.message.edit_text(
        "📋 *مدیریت پیش‌آزمون‌ها*\n\n"
        f"تعداد پیش‌آزمون: {len(preexams)}\n\n"
        "یک پیش‌آزمون را انتخاب کنید یا یکی جدید ایجاد کنید:",
        parse_mode="Markdown",
        reply_markup=InlineKeyboardMarkup(keyboard)
    )


@admin_only
async def create_preexam_start(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """شروع ایجاد پیش‌آزمون جدید."""
    query = update.callback_query
    await query.answer()
    
    # ایجاد پیش‌آزمون با عنوان پیش‌فرض
    preexams = await PreexamRepository.get_all_active()
    new_num = len(preexams) + 1
    title = f"پیش آزمون مرحله {new_num}"
    
    preexam_id = await PreexamRepository.create(
        title=title,
        description=None,
        created_by=update.effective_user.id
    )
    
    await query.message.edit_text(
        f"✅ پیش‌آزمون «{title}» ایجاد شد.\n\n"
        f"🆔 شناسه: {preexam_id}\n\n"
        "اکنون می‌توانید سوالات را به آن اضافه کنید.",
        reply_markup=InlineKeyboardMarkup([
            [InlineKeyboardButton("➕ افزودن سوال", callback_data=f"preexam_add_q_{preexam_id}")],
            [InlineKeyboardButton("🔙 بازگشت", callback_data="admin_preexams")]
        ])
    )


@admin_only
async def select_preexam_to_manage(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """انتخاب پیش‌آزمون برای مدیریت."""
    query = update.callback_query
    await query.answer()
    
    preexam_id = int(query.data.split("_")[2])
    preexam = await PreexamRepository.get_by_id(preexam_id)
    
    if not preexam:
        await query.answer("❌ پیش‌آزمون یافت نشد", show_alert=True)
        return
    
    q_count = await PreexamRepository.get_question_count(preexam_id)
    
    keyboard = [
        [InlineKeyboardButton("➕ افزودن سوال", callback_data=f"preexam_add_q_{preexam_id}")],
        [InlineKeyboardButton("👁 مشاهده سوالات", callback_data=f"preexam_view_q_{preexam_id}")],
        [InlineKeyboardButton("🗑 حذف پیش‌آزمون", callback_data=f"preexam_delete_{preexam_id}")],
        [InlineKeyboardButton("🔙 بازگشت", callback_data="admin_preexams")],
    ]
    
    await query.message.edit_text(
        f"📋 *{preexam['title']}*\n\n"
        f"تعداد سوالات: {q_count}\n\n"
        "عملیات مورد نظر را انتخاب کنید:",
        parse_mode="Markdown",
        reply_markup=InlineKeyboardMarkup(keyboard)
    )


@admin_only
async def add_question_to_preexam_start(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """نمایش لیست سوالات برای افزودن به پیش‌آزمون."""
    query = update.callback_query
    await query.answer()
    
    preexam_id = int(query.data.split("_")[3])
    
    # دریافت سوالات موجود
    all_questions = await QuestionRepository.get_all_active()
    existing_questions = await PreexamRepository.get_questions(preexam_id)
    existing_ids = {q["id"] for q in existing_questions}
    
    # فیلتر سوالاتی که هنوز اضافه نشده‌اند
    available = [q for q in all_questions if q.id not in existing_ids]
    
    if not available:
        await query.message.edit_text(
            "❌ سوالی برای افزودن وجود ندارد.\n\n"
            "ابتدا از منوی اصلی سوالات جدید اضافه کنید.",
            reply_markup=InlineKeyboardMarkup([[
                InlineKeyboardButton("🔙 بازگشت", callback_data=f"admin_preexam_{preexam_id}")
            ]])
        )
        return
    
    # نمایش ۱۰ سوال اول
    keyboard = []
    for q in available[:10]:
        keyboard.append([InlineKeyboardButton(
            f"📝 {q.id}: {q.text[:30]}...",
            callback_data=f"preexam_addq_{preexam_id}_{q.id}"
        )])
    
    keyboard.append([InlineKeyboardButton("🔙 بازگشت", callback_data=f"admin_preexam_{preexam_id}")])
    
    await query.message.edit_text(
        f"➕ *افزودن سوال به پیش‌آزمون*\n\n"
        f"تعداد سوالات موجود: {len(available)}\n\n"
        "یکی را انتخاب کنید:",
        parse_mode="Markdown",
        reply_markup=InlineKeyboardMarkup(keyboard)
    )


@admin_only
async def add_question_to_preexam(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """افزودن سوال به پیش‌آزمون."""
    query = update.callback_query
    await query.answer()
    
    parts = query.data.split("_")
    preexam_id = int(parts[2])
    question_id = int(parts[3])
    
    # دریافت تعداد فعلی برای order
    q_count = await PreexamRepository.get_question_count(preexam_id)
    
    success = await PreexamRepository.add_question(preexam_id, question_id, q_count + 1)
    
    if success:
        new_count = await PreexamRepository.get_question_count(preexam_id)
        await query.message.edit_text(
            f"✅ سوال {question_id} به پیش‌آزمون اضافه شد.\n\n"
            f"تعداد سوالات: {new_count}",
            reply_markup=InlineKeyboardMarkup([
                [InlineKeyboardButton("➕ افزودن سوال دیگر", callback_data=f"preexam_add_q_{preexam_id}")],
                [InlineKeyboardButton("🔙 بازگشت", callback_data=f"admin_preexam_{preexam_id}")]
            ])
        )
    else:
        await query.answer("❌ خطا در افزودن سوال", show_alert=True)


@admin_only
async def view_preexam_questions(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """مشاهده سوالات پیش‌آزمون."""
    query = update.callback_query
    await query.answer()
    
    preexam_id = int(query.data.split("_")[3])
    questions = await PreexamRepository.get_questions(preexam_id)
    preexam = await PreexamRepository.get_by_id(preexam_id)
    
    if not questions:
        await query.message.edit_text(
            f"📋 *{preexam['title']}*\n\n"
            "هیچ سوالی در این پیش‌آزمون نیست.",
            parse_mode="Markdown",
            reply_markup=InlineKeyboardMarkup([[
                InlineKeyboardButton("➕ افزودن سوال", callback_data=f"preexam_add_q_{preexam_id}"),
                InlineKeyboardButton("🔙 بازگشت", callback_data=f"admin_preexam_{preexam_id}")
            ]])
        )
        return
    
    text = f"📋 *{preexam['title']}*\n\n*سوالات:*\n\n"
    for i, q in enumerate(questions, 1):
        correct = {"ا": "الف", "B": "ب", "C": "ج", "D": "د"}.get(q["correct_answer"], q["correct_answer"])
        text += f"{i}. {q['text'][:40]}... (پاسخ: {correct})\n"
    
    await query.message.edit_text(
        text,
        parse_mode="Markdown",
        reply_markup=InlineKeyboardMarkup([[
            InlineKeyboardButton("🔙 بازگشت", callback_data=f"admin_preexam_{preexam_id}")
        ]])
    )


@admin_only
async def delete_preexam(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """حذف پیش‌آزمون."""
    query = update.callback_query
    await query.answer()
    
    preexam_id = int(query.data.split("_")[2])
    
    await PreexamRepository.delete(preexam_id)
    
    await query.message.edit_text(
        "✅ پیش‌آزمون حذف شد.",
        reply_markup=InlineKeyboardMarkup([[
            InlineKeyboardButton("🔙 بازگشت", callback_data="admin_preexams")
        ]])
    )


# ==================== هندلرهای مخزن سوالات ====================

@admin_only
async def show_questions_repo(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """نمایش مخزن سوالات آزمون."""
    query = update.callback_query
    await query.answer()
    
    questions = await QuestionRepository.get_all_active()
    
    if not questions:
        await query.message.edit_text(
            "📦 *مخزن سوالات آزمون*\n\n"
            "هیچ سوالی در مخزن وجود ندارد.",
            parse_mode="Markdown",
            reply_markup=InlineKeyboardMarkup([[
                InlineKeyboardButton("🔙 بازگشت", callback_data="back_to_admin_menu")
            ]])
        )
        return
    
    # نمایش 10 سوال آخر
    keyboard = []
    for q in questions[-10:]:
        keyboard.append([InlineKeyboardButton(
            f"🗑 {q.id}: {q.text[:30]}...",
            callback_data=f"del_q_{q.id}"
        )])
    
    keyboard.append([InlineKeyboardButton("🔙 بازگشت", callback_data="back_to_admin_menu")])
    
    await query.message.edit_text(
        f"📦 *مخزن سوالات آزمون*\n\n"
        f"تعداد: {len(questions)} سوال\n\n"
        "برای حذف روی سوال کلیک کنید:",
        parse_mode="Markdown",
        reply_markup=InlineKeyboardMarkup(keyboard)
    )


@admin_only
async def show_lucky_repo(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """نمایش مخزن تست شانسی."""
    query = update.callback_query
    await query.answer()
    
    from database import Database
    conn = await Database.get_connection()
    cursor = await conn.execute(
        "SELECT * FROM user_questions WHERE is_active = 1 ORDER BY id DESC LIMIT 10"
    )
    questions = await cursor.fetchall()
    
    cursor = await conn.execute(
        "SELECT COUNT(*) as count FROM user_questions WHERE is_active = 1"
    )
    row = await cursor.fetchone()
    total = row["count"] if row else 0
    
    if not questions:
        await query.message.edit_text(
            "🎲 *مخزن تست شانسی*\n\n"
            "هیچ سوالی در مخزن وجود ندارد.",
            parse_mode="Markdown",
            reply_markup=InlineKeyboardMarkup([[
                InlineKeyboardButton("🔙 بازگشت", callback_data="back_to_admin_menu")
            ]])
        )
        return
    
    keyboard = []
    for q in questions:
        keyboard.append([InlineKeyboardButton(
            f"🗑 {q['id']}: {q['text'][:30]}...",
            callback_data=f"del_lucky_{q['id']}"
        )])
    
    keyboard.append([InlineKeyboardButton("🔙 بازگشت", callback_data="back_to_admin_menu")])
    
    await query.message.edit_text(
        f"🎲 *مخزن تست شانسی*\n\n"
        f"تعداد: {total} سوال\n\n"
        "برای حذف روی سوال کلیک کنید:",
        parse_mode="Markdown",
        reply_markup=InlineKeyboardMarkup(keyboard)
    )


@admin_only
async def delete_question_from_repo(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """حذف سوال از مخزن آزمون."""
    query = update.callback_query
    await query.answer()
    
    question_id = int(query.data.split("_")[2])
    
    await QuestionRepository.delete(question_id)
    
    await query.message.edit_text(
        f"✅ سوال {question_id} از مخزن آزمون حذف شد.",
        reply_markup=InlineKeyboardMarkup([
            [InlineKeyboardButton("📦 برگشت به مخزن", callback_data="admin_questions_repo")],
            [InlineKeyboardButton("🔙 پنل ادمین", callback_data="back_to_admin_menu")]
        ])
    )


@admin_only
async def delete_lucky_question(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """حذف سوال از مخزن تست شانسی."""
    query = update.callback_query
    await query.answer()
    
    question_id = int(query.data.split("_")[2])
    
    from database import Database
    conn = await Database.get_connection()
    await conn.execute(
        "UPDATE user_questions SET is_active = 0 WHERE id = ?",
        (question_id,)
    )
    await conn.commit()
    
    await query.message.edit_text(
        f"✅ سوال {question_id} از مخزن تست شانسی حذف شد.",
        reply_markup=InlineKeyboardMarkup([
            [InlineKeyboardButton("🎲 برگشت به مخزن", callback_data="admin_lucky_repo")],
            [InlineKeyboardButton("🔙 پنل ادمین", callback_data="back_to_admin_menu")]
        ])
    )
