# handlers/anonymous.py - هندلر پیام‌های ناشناس

from telegram import (
    Update,
    InlineKeyboardButton,
    InlineKeyboardMarkup,
    ReplyKeyboardMarkup,
    ReplyKeyboardRemove,
)
from telegram.ext import (
    ContextTypes,
    ConversationHandler,
    MessageHandler,
    CallbackQueryHandler,
    filters,
)

from config import ADMIN_IDS
from database import Database
from repositories import UserRepository
from handlers.auth import get_main_menu_keyboard


# State
WAITING_MESSAGE = 1
WAITING_REPLY = 2


# ==================== Repository ====================

class AnonymousMessageRepository:
    """Repository برای پیام‌های ناشناس."""

    @staticmethod
    async def insert(sender_id: int, message_text: str) -> int:
        """ذخیره پیام جدید."""
        conn = await Database.get_connection()
        cursor = await conn.execute(
            """INSERT INTO anonymous_messages (sender_id, message_text)
               VALUES (?, ?)""",
            (sender_id, message_text)
        )
        await conn.commit()
        return cursor.lastrowid

    @staticmethod
    async def get_unread() -> list:
        """دریافت پیام‌های خوانده نشده."""
        conn = await Database.get_connection()
        cursor = await conn.execute(
            """SELECT am.*, u.full_name as sender_name 
               FROM anonymous_messages am
               LEFT JOIN users u ON am.sender_id = u.telegram_id
               WHERE am.is_read = 0 AND am.reply_text IS NULL
               ORDER BY am.created_at DESC"""
        )
        rows = await cursor.fetchall()
        return [dict(row) for row in rows]

    @staticmethod
    async def get_by_id(msg_id: int) -> dict:
        """دریافت پیام با شناسه."""
        conn = await Database.get_connection()
        cursor = await conn.execute(
            "SELECT * FROM anonymous_messages WHERE id = ?",
            (msg_id,)
        )
        row = await cursor.fetchone()
        return dict(row) if row else None

    @staticmethod
    async def mark_as_read(msg_id: int) -> None:
        """علامت‌گذاری به عنوان خوانده شده."""
        conn = await Database.get_connection()
        await conn.execute(
            "UPDATE anonymous_messages SET is_read = 1 WHERE id = ?",
            (msg_id,)
        )
        await conn.commit()

    @staticmethod
    async def reply(msg_id: int, reply_text: str, replied_by: int) -> None:
        """ذخیره پاسخ."""
        conn = await Database.get_connection()
        await conn.execute(
            """UPDATE anonymous_messages 
               SET reply_text = ?, replied_by = ?, replied_at = CURRENT_TIMESTAMP, is_read = 1
               WHERE id = ?""",
            (reply_text, replied_by, msg_id)
        )
        await conn.commit()

    @staticmethod
    async def get_unread_count() -> int:
        """دریافت تعداد پیام‌های خوانده نشده."""
        conn = await Database.get_connection()
        cursor = await conn.execute(
            "SELECT COUNT(*) as count FROM anonymous_messages WHERE is_read = 0"
        )
        row = await cursor.fetchone()
        return row["count"] if row else 0

    @staticmethod
    async def delete(msg_id: int) -> None:
        """حذف پیام."""
        conn = await Database.get_connection()
        await conn.execute(
            "DELETE FROM anonymous_messages WHERE id = ?",
            (msg_id,)
        )
        await conn.commit()


# ==================== کیبوردها ====================

def get_cancel_keyboard() -> ReplyKeyboardMarkup:
    """کیبورد انصراف."""
    return ReplyKeyboardMarkup(
        [["❌ انصراف"]],
        resize_keyboard=True,
        one_time_keyboard=True
    )


# ==================== هندلرهای کاربر ====================

async def start_anonymous_message(update: Update, context: ContextTypes.DEFAULT_TYPE) -> int:
    """شروع ارسال پیام ناشناس."""
    user_id = update.effective_user.id
    
    # چک ثبت‌نام
    db_user = await UserRepository.get_by_telegram_id(user_id)
    if not db_user or not db_user.is_registration_complete():
        await update.message.reply_text(
            "❌ لطفاً ابتدا ثبت‌نام را تکمیل کنید. دستور /start را ارسال کنید."
        )
        return ConversationHandler.END
    
    await update.message.reply_text(
        "📨 *ارسال پیام ناشناس*\n\n"
        "💬 پیام خود را برای ارسال به صورت ناشناس به مدیریت ارسال کنید:\n\n"
        "_پیام شما به صورت ناشناس ارسال خواهد شد._",
        parse_mode="Markdown",
        reply_markup=get_cancel_keyboard()
    )
    return WAITING_MESSAGE


async def receive_anonymous_message(update: Update, context: ContextTypes.DEFAULT_TYPE) -> int:
    """دریافت پیام ناشناس."""
    text = update.message.text.strip()
    
    if text == "❌ انصراف":
        await update.message.reply_text(
            "❌ ارسال پیام لغو شد.",
            reply_markup=get_main_menu_keyboard()
        )
        return ConversationHandler.END
    
    if len(text) < 3:
        await update.message.reply_text("❌ پیام خیلی کوتاه است. لطفاً پیام بلندتری ارسال کنید:")
        return WAITING_MESSAGE
    
    user_id = update.effective_user.id
    
    # ذخیره پیام
    msg_id = await AnonymousMessageRepository.insert(user_id, text)
    
    await update.message.reply_text(
        "✅ پیام شما با موفقیت ارسال شد.\n\n"
        "_در صورت پاسخ مدیریت، پیام برای شما ارسال خواهد شد._",
        parse_mode="Markdown",
        reply_markup=get_main_menu_keyboard()
    )
    
    # اطلاع به ادمین‌ها
    unread_count = await AnonymousMessageRepository.get_unread_count()
    for admin_id in ADMIN_IDS:
        try:
            await context.bot.send_message(
                chat_id=admin_id,
                text=f"📨 *پیام ناشناس جدید!*\n\n"
                     f"🆔 شناسه: {msg_id}\n"
                     f"📝 متن: {text[:100]}{'...' if len(text) > 100 else ''}\n\n"
                     f"📬 پیام‌های خوانده نشده: {unread_count}",
                parse_mode="Markdown",
                reply_markup=InlineKeyboardMarkup([[
                    InlineKeyboardButton("📨 مدیریت پیام‌ها", callback_data="admin_anon_messages")
                ]])
            )
        except:
            pass
    
    return ConversationHandler.END


async def cancel_anonymous(update: Update, context: ContextTypes.DEFAULT_TYPE) -> int:
    """انصراف از ارسال پیام."""
    await update.message.reply_text(
        "❌ ارسال پیام لغو شد.",
        reply_markup=get_main_menu_keyboard()
    )
    return ConversationHandler.END


# ==================== هندلرهای ادمین ====================

async def show_anonymous_messages(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """نمایش پیام‌های ناشناس برای ادمین."""
    query = update.callback_query
    await query.answer()
    
    user_id = update.effective_user.id
    if user_id not in ADMIN_IDS:
        await query.answer("❌ شما دسترسی ندارید!", show_alert=True)
        return
    
    messages = await AnonymousMessageRepository.get_unread()
    
    if not messages:
        await query.message.edit_text(
            "📨 *پیام‌های ناشناس*\n\n"
            "هیچ پیام خوانده نشده‌ای وجود ندارد.",
            parse_mode="Markdown",
            reply_markup=InlineKeyboardMarkup([[
                InlineKeyboardButton("🔙 بازگشت", callback_data="back_to_admin_menu")
            ]])
        )
        return
    
    # ذخیره لیست پیام‌ها
    context.user_data["anon_messages"] = messages
    context.user_data["anon_index"] = 0
    
    await show_anon_message(query, context, 0)


async def show_anon_message(query, context: ContextTypes.DEFAULT_TYPE, index: int) -> None:
    """نمایش یک پیام ناشناس."""
    messages = context.user_data.get("anon_messages", [])
    
    if not messages or index >= len(messages):
        return
    
    msg = messages[index]
    
    # تبدیل تاریخ به فرمت فارسی
    created = msg['created_at'][:16].replace('-', '/').replace('T', ' - ')
    
    text = f"""📨 پیام ناشناس

🕐 {created}

{msg['message_text']}"""
    
    keyboard = [
        [InlineKeyboardButton("💬 پاسخ دادن", callback_data=f"anon_reply_{msg['id']}")],
        [InlineKeyboardButton("✅ خوانده شد", callback_data=f"anon_read_{msg['id']}")],
        [InlineKeyboardButton("🗑 حذف", callback_data=f"anon_del_{msg['id']}")],
    ]
    
    # ناویگیشن
    nav_row = []
    if index > 0:
        nav_row.append(InlineKeyboardButton("⬅️ قبلی", callback_data="anon_prev"))
    nav_row.append(InlineKeyboardButton(f"{index + 1}/{len(messages)}", callback_data="none"))
    if index < len(messages) - 1:
        nav_row.append(InlineKeyboardButton("➡️ بعدی", callback_data="anon_next"))
    keyboard.append(nav_row)
    
    keyboard.append([InlineKeyboardButton("🔙 بازگشت", callback_data="back_to_admin_menu")])
    
    await query.message.edit_text(
        text,
        reply_markup=InlineKeyboardMarkup(keyboard)
    )


async def anon_nav(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """ناویگیشن در پیام‌ها."""
    query = update.callback_query
    await query.answer()
    
    messages = context.user_data.get("anon_messages", [])
    current_index = context.user_data.get("anon_index", 0)
    
    if query.data == "anon_next":
        current_index = min(current_index + 1, len(messages) - 1)
    elif query.data == "anon_prev":
        current_index = max(current_index - 1, 0)
    
    context.user_data["anon_index"] = current_index
    await show_anon_message(query, context, current_index)


async def start_reply_to_anon(update: Update, context: ContextTypes.DEFAULT_TYPE) -> int:
    """شروع پاسخ به پیام ناشناس."""
    query = update.callback_query
    await query.answer()
    
    user_id = update.effective_user.id
    if user_id not in ADMIN_IDS:
        return ConversationHandler.END
    
    msg_id = int(query.data.split("_")[2])
    msg = await AnonymousMessageRepository.get_by_id(msg_id)
    
    if not msg:
        await query.answer("❌ پیام یافت نشد", show_alert=True)
        return ConversationHandler.END
    
    context.user_data["replying_to_anon"] = msg_id
    context.user_data["replying_to_sender"] = msg["sender_id"]
    
    await query.message.edit_text(
        f"💬 *پاسخ به پیام ناشناس*\n\n"
        f"📝 پیام اصلی:\n_{msg['message_text']}_\n\n"
        f"لطفاً پاسخ خود را ارسال کنید:",
        parse_mode="Markdown"
    )
    
    await query.message.reply_text(
        "پاسخ را تایپ کنید:",
        reply_markup=get_cancel_keyboard()
    )
    
    return WAITING_REPLY


async def receive_reply(update: Update, context: ContextTypes.DEFAULT_TYPE) -> int:
    """دریافت پاسخ ادمین."""
    text = update.message.text.strip()
    
    if text == "❌ انصراف":
        await update.message.reply_text(
            "❌ پاسخ لغو شد.",
            reply_markup=ReplyKeyboardRemove()
        )
        return ConversationHandler.END
    
    msg_id = context.user_data.get("replying_to_anon")
    sender_id = context.user_data.get("replying_to_sender")
    admin_id = update.effective_user.id
    
    if not msg_id or not sender_id:
        await update.message.reply_text("❌ خطا در ارسال پاسخ.")
        return ConversationHandler.END
    
    # ذخیره پاسخ
    await AnonymousMessageRepository.reply(msg_id, text, admin_id)
    
    # ارسال پاسخ به کاربر
    try:
        await context.bot.send_message(
            chat_id=sender_id,
            text=f"📬 *پاسخ به پیام ناشناس شما:*\n\n"
                 f"💬 {text}",
            parse_mode="Markdown"
        )
        await update.message.reply_text(
            "✅ پاسخ ارسال شد.",
            reply_markup=ReplyKeyboardRemove()
        )
    except Exception as e:
        await update.message.reply_text(
            f"⚠️ پاسخ ذخیره شد ولی ارسال به کاربر ناموفق بود.\n\n{str(e)}",
            reply_markup=ReplyKeyboardRemove()
        )
    
    # پاک کردن داده موقت
    context.user_data.pop("replying_to_anon", None)
    context.user_data.pop("replying_to_sender", None)
    
    return ConversationHandler.END


async def mark_anon_read(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """علامت‌گذاری پیام به عنوان خوانده شده."""
    query = update.callback_query
    await query.answer()
    
    msg_id = int(query.data.split("_")[2])
    await AnonymousMessageRepository.mark_as_read(msg_id)
    
    await query.message.edit_text(
        "✅ پیام به عنوان خوانده شده علامت‌گذاری شد.",
        reply_markup=InlineKeyboardMarkup([[
            InlineKeyboardButton("📨 پیام‌های دیگر", callback_data="admin_anon_messages"),
            InlineKeyboardButton("🔙 پنل ادمین", callback_data="back_to_admin_menu")
        ]])
    )


async def delete_anon_message(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """حذف پیام ناشناس."""
    query = update.callback_query
    await query.answer()
    
    msg_id = int(query.data.split("_")[2])
    await AnonymousMessageRepository.delete(msg_id)
    
    await query.message.edit_text(
        "✅ پیام حذف شد.",
        reply_markup=InlineKeyboardMarkup([[
            InlineKeyboardButton("📨 پیام‌های دیگر", callback_data="admin_anon_messages"),
            InlineKeyboardButton("🔙 پنل ادمین", callback_data="back_to_admin_menu")
        ]])
    )


async def cancel_reply(update: Update, context: ContextTypes.DEFAULT_TYPE) -> int:
    """انصراف از پاسخ."""
    context.user_data.pop("replying_to_anon", None)
    context.user_data.pop("replying_to_sender", None)
    
    await update.message.reply_text(
        "❌ پاسخ لغو شد.",
        reply_markup=ReplyKeyboardRemove()
    )
    return ConversationHandler.END


# ==================== تابع دریافت هندلرها ====================

def get_anonymous_handlers() -> list:
    """برگرداندن لیست هندلرهای پیام ناشناس."""
    
    # ConversationHandler برای ارسال پیام ناشناس (کاربر)
    send_message_handler = ConversationHandler(
        entry_points=[
            MessageHandler(filters.Regex("^📨 ارسال پیام ناشناس$"), start_anonymous_message),
        ],
        states={
            WAITING_MESSAGE: [
                MessageHandler(filters.TEXT & ~filters.COMMAND, receive_anonymous_message),
            ],
        },
        fallbacks=[
            MessageHandler(filters.Regex("^❌ انصراف$"), cancel_anonymous),
        ],
        allow_reentry=True,
    )
    
    # ConversationHandler برای پاسخ ادمین
    reply_handler = ConversationHandler(
        entry_points=[
            CallbackQueryHandler(start_reply_to_anon, pattern=r"^anon_reply_\d+$"),
        ],
        states={
            WAITING_REPLY: [
                MessageHandler(filters.TEXT & ~filters.COMMAND, receive_reply),
            ],
        },
        fallbacks=[
            MessageHandler(filters.Regex("^❌ انصراف$"), cancel_reply),
        ],
        allow_reentry=True,
    )
    
    return [
        send_message_handler,
        reply_handler,
        CallbackQueryHandler(show_anonymous_messages, pattern="^admin_anon_messages$"),
        CallbackQueryHandler(anon_nav, pattern="^anon_(next|prev)$"),
        CallbackQueryHandler(mark_anon_read, pattern=r"^anon_read_\d+$"),
        CallbackQueryHandler(delete_anon_message, pattern=r"^anon_del_\d+$"),
    ]
