# handlers/auth.py - هندلرهای احراز هویت و چک عضویت

from telegram import (
    Update,
    InlineKeyboardButton,
    InlineKeyboardMarkup,
    KeyboardButton,
    ReplyKeyboardMarkup,
    ReplyKeyboardRemove,
)
from telegram.ext import (
    ContextTypes,
    CommandHandler,
    CallbackQueryHandler,
    MessageHandler,
    filters,
)

from config import REQUIRED_CHANNEL_LINK
from services import MembershipService, ValidationService, TextService
from repositories import UserRepository


# ==================== کیبوردها ====================

def get_join_channel_keyboard() -> InlineKeyboardMarkup:
    """کیبورد شیشه‌ای عضویت در کانال."""
    keyboard = [
        [InlineKeyboardButton("📢 عضویت در کانال تراز", url=REQUIRED_CHANNEL_LINK)],
        [InlineKeyboardButton("✅ عضو شدم", callback_data="check_membership")],
    ]
    return InlineKeyboardMarkup(keyboard)


def get_phone_keyboard() -> ReplyKeyboardMarkup:
    """کیبورد ارسال شماره تماس."""
    keyboard = [
        [KeyboardButton("📱 ارسال شماره تماس", request_contact=True)],
    ]
    return ReplyKeyboardMarkup(
        keyboard,
        resize_keyboard=True,
        one_time_keyboard=True
    )


def get_main_menu_keyboard() -> ReplyKeyboardMarkup:
    """کیبورد منوی اصلی."""
    keyboard = [
        [KeyboardButton("🎯 آزمون تراز | رایگان")],
        [KeyboardButton("🛍 خدمات و محصولات")],
        [KeyboardButton("👤 پروفایل"), KeyboardButton("📊 نتایج من")],
        [KeyboardButton("📨 ارسال پیام ناشناس"), KeyboardButton("📞 پشتیبانی")],
        [KeyboardButton("ℹ️ معرفی مجموعه")],
    ]
    return ReplyKeyboardMarkup(keyboard, resize_keyboard=True)



def get_start_registration_keyboard() -> InlineKeyboardMarkup:
    """کیبورد شیشه‌ای شروع ثبت‌نام."""
    keyboard = [
        [InlineKeyboardButton("📝 شروع ثبت‌نام", callback_data="start_registration")],
    ]
    return InlineKeyboardMarkup(keyboard)


# ==================== هندلرها ====================

async def start_command(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """هندلر دستور /start - نقطه ورود اصلی."""
    user = update.effective_user
    
    # چک دیپ لینک پیش‌آزمون
    deep_link_preexam = None
    if context.args and len(context.args) > 0:
        arg = context.args[0]
        if arg.startswith("preexam_"):
            try:
                deep_link_preexam = int(arg.split("_")[1])
                context.user_data["deep_link_preexam"] = deep_link_preexam
            except (ValueError, IndexError):
                pass
    
    # مرحله ۱: چک عضویت کانال
    is_member = await MembershipService.check_channel_membership(
        context.bot, user.id
    )
    
    if not is_member:
        await update.message.reply_text(
            TextService.MEMBERSHIP_REQUIRED,
            parse_mode="Markdown",
            reply_markup=get_join_channel_keyboard()
        )
        return
    
    # مرحله ۲: چک کاربر در دیتابیس
    db_user = await UserRepository.get_by_telegram_id(user.id)
    
    if db_user is None:
        # کاربر جدید - ایجاد رکورد
        await UserRepository.insert(user.id)
        db_user = await UserRepository.get_by_telegram_id(user.id)
    
    # مرحله ۳: چک شماره تماس
    if not db_user.phone_number:
        await update.message.reply_text(
            TextService.PHONE_REQUEST,
            parse_mode="Markdown",
            reply_markup=get_phone_keyboard()
        )
        return
    
    # مرحله ۴: چک تکمیل ثبت‌نام
    if not db_user.is_registration_complete():
        # ذخیره وضعیت در user_data برای شروع ConversationHandler
        context.user_data["awaiting_registration"] = True
        await update.message.reply_text(
            TextService.REGISTRATION_NAME,
            parse_mode="Markdown",
            reply_markup=ReplyKeyboardRemove()
        )
        return
    
    # اگر دیپ لینک پیش‌آزمون داریم، مستقیم به پیش‌آزمون ببر
    if deep_link_preexam:
        from repositories import PreexamRepository
        preexam = await PreexamRepository.get_by_id(deep_link_preexam)
        if preexam and preexam.get('is_active', 1):
            # ذخیره و هدایت به پیش‌آزمون
            context.user_data["current_preexam_id"] = deep_link_preexam
            context.user_data["preexam_answers"] = {}
            context.user_data["preexam_current_q"] = 0
            
            # دریافت سوالات پیش‌آزمون
            questions = await PreexamRepository.get_questions(deep_link_preexam)
            if questions:
                context.user_data["preexam_questions"] = questions
                
                # نمایش اولین سوال
                q = questions[0]
                keyboard = [
                    [InlineKeyboardButton("الف", callback_data=f"preexam_ans_A"),
                     InlineKeyboardButton("ب", callback_data=f"preexam_ans_B")],
                    [InlineKeyboardButton("ج", callback_data=f"preexam_ans_C"),
                     InlineKeyboardButton("د", callback_data=f"preexam_ans_D")],
                ]
                
                await update.message.reply_text(
                    f"📋 *{preexam['title']}*\n\n"
                    f"سوال ۱ از {len(questions)}:\n\n"
                    f"{q['text']}\n\n"
                    f"*الف)* {q['option_a']}\n"
                    f"*ب)* {q['option_b']}\n"
                    f"*ج)* {q['option_c']}\n"
                    f"*د)* {q['option_d']}",
                    parse_mode="Markdown",
                    reply_markup=InlineKeyboardMarkup(keyboard)
                )
                return
    
    # همه چیز اوکی - نمایش منوی اصلی
    await update.message.reply_text(
        TextService.MAIN_MENU,
        parse_mode="Markdown",
        reply_markup=get_main_menu_keyboard()
    )



async def check_membership_callback(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """هندلر کالبک چک عضویت کانال."""
    query = update.callback_query
    await query.answer()
    
    user = update.effective_user
    
    is_member = await MembershipService.check_channel_membership(
        context.bot, user.id
    )
    
    if not is_member:
        await query.answer(
            "❌ هنوز عضو کانال نشدید! لطفاً ابتدا عضو شوید.",
            show_alert=True
        )
        return
    
    # عضویت تایید شد - ادامه فرآیند
    await query.message.edit_text(
        "✅ عضویت شما تایید شد!",
        parse_mode="Markdown"
    )
    
    # چک کاربر در دیتابیس
    db_user = await UserRepository.get_by_telegram_id(user.id)
    
    if db_user is None:
        await UserRepository.insert(user.id)
        db_user = await UserRepository.get_by_telegram_id(user.id)
    
    # درخواست شماره تماس
    if not db_user.phone_number:
        await query.message.reply_text(
            TextService.PHONE_REQUEST,
            parse_mode="Markdown",
            reply_markup=get_phone_keyboard()
        )
        return
    
    # چک تکمیل ثبت‌نام
    if not db_user.is_registration_complete():
        await query.message.reply_text(
            "✅ عالی! حالا برای تکمیل ثبت‌نام روی دکمه زیر کلیک کنید:",
            reply_markup=get_start_registration_keyboard()
        )
        return
    
    # نمایش منوی اصلی
    await query.message.reply_text(
        TextService.MAIN_MENU,
        parse_mode="Markdown",
        reply_markup=get_main_menu_keyboard()
    )


async def contact_received(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """هندلر دریافت شماره تماس."""
    contact = update.message.contact
    user = update.effective_user
    
    # اطمینان از اینکه شماره متعلق به خود کاربر است
    if contact.user_id != user.id:
        await update.message.reply_text(
            "❌ لطفاً شماره تماس خودتان را ارسال کنید!",
            reply_markup=get_phone_keyboard()
        )
        return
    
    phone_number = contact.phone_number
    
    # نرمال‌سازی شماره
    normalized_phone = ValidationService.normalize_phone(phone_number)
    
    # ذخیره در دیتابیس
    db_user = await UserRepository.get_by_telegram_id(user.id)
    
    if db_user is None:
        await UserRepository.insert(user.id, normalized_phone)
    else:
        await UserRepository.update_phone(user.id, normalized_phone)
    
    # بررسی وضعیت ثبت‌نام
    db_user = await UserRepository.get_by_telegram_id(user.id)
    
    if not db_user.is_registration_complete():
        # نمایش دکمه شروع ثبت‌نام
        await update.message.reply_text(
            f"✅ شماره {normalized_phone} ذخیره شد.\n\nبرای تکمیل ثبت‌نام روی دکمه زیر کلیک کنید:",
            reply_markup=get_start_registration_keyboard()
        )
        return
    
    # ثبت‌نام کامل - نمایش منوی اصلی
    await update.message.reply_text(
        TextService.MAIN_MENU,
        parse_mode="Markdown",
        reply_markup=get_main_menu_keyboard()
    )


# ==================== تابع دریافت هندلرها ====================

def get_auth_handlers() -> list:
    """برگرداندن لیست هندلرهای احراز هویت."""
    return [
        CommandHandler("start", start_command),
        CallbackQueryHandler(check_membership_callback, pattern="^check_membership$"),
        MessageHandler(filters.CONTACT, contact_received),
    ]
