# handlers/exam.py - هندلر آزمون

from telegram import (
    Update,
    InlineKeyboardButton,
    InlineKeyboardMarkup,
)
from telegram.ext import (
    ContextTypes,
    MessageHandler,
    CallbackQueryHandler,
    filters,
)

from config import EXAM_QUESTION_COUNT, NEGATIVE_SCORE_FACTOR
from repositories import QuestionRepository, ExamResultRepository, UserRepository
from handlers.auth import get_main_menu_keyboard


# ==================== کیبوردها ====================

def get_question_keyboard(question_id: int) -> InlineKeyboardMarkup:
    """کیبورد گزینه‌های سوال."""
    keyboard = [
        [
            InlineKeyboardButton("الف", callback_data=f"answer_{question_id}_A"),
            InlineKeyboardButton("ب", callback_data=f"answer_{question_id}_B"),
        ],
        [
            InlineKeyboardButton("ج", callback_data=f"answer_{question_id}_C"),
            InlineKeyboardButton("د", callback_data=f"answer_{question_id}_D"),
        ],
        [
            InlineKeyboardButton("🔙 بازگشت به منو", callback_data="back_to_menu"),
        ],
    ]
    return InlineKeyboardMarkup(keyboard)


def get_exam_result_keyboard() -> InlineKeyboardMarkup:
    """کیبورد پایان آزمون."""
    keyboard = [
        [InlineKeyboardButton("📊 تحلیل سوالات", callback_data="show_analysis")],
        [InlineKeyboardButton("🔄 آزمون جدید", callback_data="new_exam")],
        [InlineKeyboardButton("🔙 بازگشت به منو", callback_data="back_to_menu")],
    ]
    return InlineKeyboardMarkup(keyboard)


def get_analysis_keyboard(answers: list) -> InlineKeyboardMarkup:
    """کیبورد تحلیل سوالات با علامت درست/غلط."""
    keyboard = []
    # نمایش سوالات در ۲ ستون
    for i in range(0, len(answers), 2):
        row = []
        # سوال اول در ردیف
        ans1 = answers[i]
        emoji1 = "✅" if ans1["is_correct"] else "❌"
        row.append(InlineKeyboardButton(
            f"{emoji1} سوال {i + 1}",
            callback_data=f"review_{i}"
        ))
        # سوال دوم در ردیف (اگر وجود دارد)
        if i + 1 < len(answers):
            ans2 = answers[i + 1]
            emoji2 = "✅" if ans2["is_correct"] else "❌"
            row.append(InlineKeyboardButton(
                f"{emoji2} سوال {i + 2}",
                callback_data=f"review_{i + 1}"
            ))
        keyboard.append(row)
    
    keyboard.append([InlineKeyboardButton("🔙 بازگشت به کارنامه", callback_data="back_to_result")])
    keyboard.append([InlineKeyboardButton("🏠 منوی اصلی", callback_data="back_to_menu")])
    return InlineKeyboardMarkup(keyboard)


# ==================== توابع کمکی ====================

def format_question(question, current_num: int, total: int) -> str:
    """فرمت‌دهی سوال برای نمایش."""
    return f"""
📝 *سوال {current_num} از {total}*

{question.text}

*الف)* {question.option_a}
*ب)* {question.option_b}
*ج)* {question.option_c}
*د)* {question.option_d}
"""


def format_result(correct: int, wrong: int, blank: int, total: int) -> str:
    """فرمت‌دهی کارنامه آزمون با نمره منفی."""
    # محاسبه درصد بدون نمره منفی
    score_without_negative = (correct / total) * 100 if total > 0 else 0
    
    # محاسبه درصد با نمره منفی (هر 3 غلط = 1 درست کم)
    effective_correct = correct - (wrong * NEGATIVE_SCORE_FACTOR)
    effective_correct = max(0, effective_correct)  # حداقل صفر
    score_with_negative = (effective_correct / total) * 100 if total > 0 else 0
    
    # تعیین ایموجی و پیام بر اساس درصد با نمره منفی
    if score_with_negative >= 80:
        emoji = "🏆"
        message = "عالی بود!"
    elif score_with_negative >= 60:
        emoji = "👍"
        message = "خوب بود!"
    elif score_with_negative >= 40:
        emoji = "📚"
        message = "بیشتر تلاش کن!"
    else:
        emoji = "💪"
        message = "نیاز به تمرین بیشتر داری!"
    
    return f"""
{emoji} *کارنامه آزمون*

✅ پاسخ‌های صحیح: {correct}
❌ پاسخ‌های غلط: {wrong}
⬜ پاسخ نداده: {blank}
📊 مجموع سوالات: {total}

━━━━━━━━━━━━━━━
📈 *درصد بدون نمره منفی:* {score_without_negative:.1f}%
📉 *درصد با نمره منفی:* {score_with_negative:.1f}%
━━━━━━━━━━━━━━━

{message}
"""


# ==================== هندلرها ====================

async def start_exam(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """شروع آزمون جدید."""
    user_id = update.effective_user.id
    
    # چک تکمیل ثبت‌نام
    db_user = await UserRepository.get_by_telegram_id(user_id)
    if not db_user or not db_user.is_registration_complete():
        await update.message.reply_text(
            "❌ لطفاً ابتدا ثبت‌نام را تکمیل کنید. دستور /start را ارسال کنید."
        )
        return
    
    # دریافت سوالات تصادفی از مخزن A
    questions = await QuestionRepository.get_random_questions(limit=EXAM_QUESTION_COUNT)
    
    if not questions:
        await update.message.reply_text(
            "❌ متاسفانه سوالی در سیستم موجود نیست.",
            reply_markup=get_main_menu_keyboard()
        )
        return
    
    # ذخیره وضعیت آزمون در user_data
    context.user_data["exam"] = {
        "questions": questions,
        "current_index": 0,
        "correct_count": 0,
        "wrong_count": 0,
        "answers": [],
    }
    
    # نمایش اولین سوال
    question = questions[0]
    await update.message.reply_text(
        format_question(question, 1, len(questions)),
        parse_mode="Markdown",
        reply_markup=get_question_keyboard(question.id)
    )


async def handle_answer(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """هندل کردن پاسخ کاربر."""
    query = update.callback_query
    await query.answer()
    
    # پارس کردن callback_data
    data = query.data.split("_")
    if len(data) != 3:
        return
    
    _, question_id, answer = data
    question_id = int(question_id)
    
    # چک وضعیت آزمون
    exam = context.user_data.get("exam")
    if not exam:
        await query.message.edit_text(
            "❌ آزمونی در جریان نیست. از منو آزمون جدید شروع کنید.",
            reply_markup=get_exam_result_keyboard()
        )
        return
    
    questions = exam["questions"]
    current_index = exam["current_index"]
    
    if current_index >= len(questions):
        return
    
    current_question = questions[current_index]
    
    # بررسی صحت پاسخ
    is_correct = (answer == current_question.correct_answer)
    if is_correct:
        exam["correct_count"] += 1
    else:
        exam["wrong_count"] += 1
    
    # ذخیره پاسخ
    exam["answers"].append({
        "question_id": question_id,
        "user_answer": answer,
        "correct_answer": current_question.correct_answer,
        "is_correct": is_correct,
        "explanation": current_question.explanation,
    })
    
    # فیدبک کوتاه
    if is_correct:
        feedback = "✅ صحیح!"
    else:
        correct_option = {
            "A": "الف",
            "B": "ب",
            "C": "ج",
            "D": "د",
        }.get(current_question.correct_answer, "")
        feedback = f"❌ اشتباه! پاسخ صحیح: {correct_option}"
    
    # رفتن به سوال بعدی
    exam["current_index"] += 1
    context.user_data["exam"] = exam
    
    if exam["current_index"] >= len(questions):
        # پایان آزمون
        total = len(questions)
        correct = exam["correct_count"]
        wrong = exam["wrong_count"]
        blank = total - correct - wrong
        score = (correct / total) * 100 if total > 0 else 0
        
        # ذخیره نتیجه در دیتابیس
        user_id = update.effective_user.id
        await ExamResultRepository.insert(user_id, total, correct)
        
        # ذخیره پاسخ‌ها برای تحلیل (به جای پاک کردن)
        context.user_data["last_exam_answers"] = exam["answers"]
        context.user_data["last_exam_questions"] = [(q.id, q.text, q.option_a, q.option_b, q.option_c, q.option_d, q.correct_answer, q.explanation) for q in questions]
        context.user_data["last_exam_result"] = {
            "correct": correct,
            "wrong": wrong,
            "blank": blank,
            "total": total
        }
        context.user_data.pop("exam", None)
        
        await query.message.edit_text(
            f"{feedback}\n\n" + format_result(correct, wrong, blank, total),
            parse_mode="Markdown",
            reply_markup=get_exam_result_keyboard()
        )
    else:
        # نمایش سوال بعدی
        next_question = questions[exam["current_index"]]
        await query.message.edit_text(
            f"{feedback}\n\n" + format_question(
                next_question,
                exam["current_index"] + 1,
                len(questions)
            ),
            parse_mode="Markdown",
            reply_markup=get_question_keyboard(next_question.id)
        )


async def back_to_menu(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """بازگشت به منوی اصلی."""
    query = update.callback_query
    await query.answer()
    
    # پاک کردن وضعیت آزمون
    context.user_data.pop("exam", None)
    
    await query.message.edit_text("🔙 به منوی اصلی برگشتید.")
    await query.message.reply_text(
        "🏠 *منوی اصلی*\n\nاز گزینه‌های زیر انتخاب کنید:",
        parse_mode="Markdown",
        reply_markup=get_main_menu_keyboard()
    )


async def new_exam_callback(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """شروع آزمون جدید از کالبک."""
    query = update.callback_query
    await query.answer()
    
    user_id = update.effective_user.id
    
    # دریافت سوالات تصادفی
    questions = await QuestionRepository.get_random_questions(limit=EXAM_QUESTION_COUNT)
    
    if not questions:
        await query.message.edit_text(
            "❌ متاسفانه سوالی در سیستم موجود نیست."
        )
        return
    
    # ذخیره وضعیت آزمون
    context.user_data["exam"] = {
        "questions": questions,
        "current_index": 0,
        "correct_count": 0,
        "wrong_count": 0,
        "answers": [],
    }
    
    # نمایش اولین سوال
    question = questions[0]
    await query.message.edit_text(
        format_question(question, 1, len(questions)),
        parse_mode="Markdown",
        reply_markup=get_question_keyboard(question.id)
    )


async def show_profile(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """نمایش پروفایل کاربر."""
    user = update.effective_user
    user_id = user.id
    db_user = await UserRepository.get_by_telegram_id(user_id)
    
    if not db_user:
        await update.message.reply_text(
            "❌ اطلاعاتی یافت نشد. لطفاً /start را ارسال کنید."
        )
        return
    
    # آیدی تلگرام - escape کاراکترهای خاص
    if user.username:
        escaped_username = user.username.replace("_", "\\_")
        telegram_username = f"@{escaped_username}"
    else:
        telegram_username = "آیدی ندارد"
    
    profile_text = f"""
👤 *پروفایل شما*

🆔 آیدی تلگرام: {telegram_username}
📱 شماره: {db_user.phone_number or 'ثبت نشده'}
👤 نام: {db_user.full_name or 'ثبت نشده'}
📚 رشته: {db_user.field or 'ثبت نشده'}
🎒 پایه: {db_user.grade or 'ثبت نشده'}
🏠 استان: {db_user.province or 'ثبت نشده'}
"""
    
    await update.message.reply_text(
        profile_text,
        parse_mode="Markdown",
        reply_markup=get_main_menu_keyboard()
    )




async def show_results(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """نمایش منوی انتخاب نوع نتایج."""
    keyboard = InlineKeyboardMarkup([
        [InlineKeyboardButton("📝 نتایج آزمون تمرینی", callback_data="results_exam")],
        [InlineKeyboardButton("📋 نتایج پیش‌آزمون", callback_data="results_preexam")],
        [InlineKeyboardButton("🔙 بازگشت", callback_data="back_to_menu")],
    ])
    
    await update.message.reply_text(
        "📊 *نتایج من*\n\n"
        "نتایج کدام آزمون را می‌خواهید مشاهده کنید؟",
        parse_mode="Markdown",
        reply_markup=keyboard
    )


async def show_exam_results(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """نمایش نتایج آزمون تمرینی."""
    query = update.callback_query
    await query.answer()
    
    user_id = update.effective_user.id
    results = await ExamResultRepository.get_user_results(user_id, limit=10)
    
    if not results:
        await query.message.edit_text(
            "📝 *نتایج آزمون تمرینی*\n\n"
            "هنوز آزمون تمرینی نداده‌اید!",
            parse_mode="Markdown",
            reply_markup=InlineKeyboardMarkup([[
                InlineKeyboardButton("🔙 بازگشت", callback_data="back_to_results_menu")
            ]])
        )
        return
    
    text = "📝 *نتایج آزمون تمرینی:*\n\n"
    
    for i, result in enumerate(results, 1):
        emoji = "🏆" if result.score >= 80 else "📊"
        text += f"{emoji} *آزمون {i}:* {result.score:.1f}% ({result.correct_answers}/{result.total_questions})\n"
    
    await query.message.edit_text(
        text,
        parse_mode="Markdown",
        reply_markup=InlineKeyboardMarkup([[
            InlineKeyboardButton("🔙 بازگشت", callback_data="back_to_results_menu")
        ]])
    )


async def show_preexam_results_menu(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """نمایش نتایج پیش‌آزمون."""
    query = update.callback_query
    await query.answer()
    
    user_id = update.effective_user.id
    
    from repositories import PreexamResultRepository, PreexamRepository
    
    # دریافت پیش‌آزمون‌های فعال
    preexams = await PreexamRepository.get_all_active()
    
    if not preexams:
        await query.message.edit_text(
            "📋 *نتایج پیش‌آزمون*\n\n"
            "هیچ پیش‌آزمونی موجود نیست.",
            parse_mode="Markdown",
            reply_markup=InlineKeyboardMarkup([[
                InlineKeyboardButton("🔙 بازگشت", callback_data="back_to_results_menu")
            ]])
        )
        return
    
    keyboard = []
    for exam in preexams:
        result = await PreexamResultRepository.get_user_result(user_id, exam["id"])
        if result:
            emoji = "🏆" if result["score_negative"] >= 80 else "📊"
            text = f"{emoji} {exam['title']} - {result['score_negative']:.1f}%"
        else:
            text = f"⏳ {exam['title']} - شرکت نکرده‌اید"
        keyboard.append([InlineKeyboardButton(text, callback_data=f"view_preexam_result_{exam['id']}")])
    
    keyboard.append([InlineKeyboardButton("🔙 بازگشت", callback_data="back_to_results_menu")])
    
    await query.message.edit_text(
        "📋 *نتایج پیش‌آزمون*\n\n"
        "یک پیش‌آزمون را انتخاب کنید:",
        parse_mode="Markdown",
        reply_markup=InlineKeyboardMarkup(keyboard)
    )


async def view_preexam_result_detail(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """نمایش جزئیات نتیجه پیش‌آزمون."""
    query = update.callback_query
    await query.answer()
    
    preexam_id = int(query.data.split("_")[3])
    user_id = update.effective_user.id
    
    from repositories import PreexamResultRepository, PreexamRepository
    
    preexam = await PreexamRepository.get_by_id(preexam_id)
    result = await PreexamResultRepository.get_user_result(user_id, preexam_id)
    
    if not result:
        await query.message.edit_text(
            f"📋 *{preexam['title']}*\n\n"
            "شما هنوز در این پیش‌آزمون شرکت نکرده‌اید.",
            parse_mode="Markdown",
            reply_markup=InlineKeyboardMarkup([[
                InlineKeyboardButton("▶️ شروع آزمون", callback_data=f"pre_start_{preexam_id}"),
                InlineKeyboardButton("🔙 بازگشت", callback_data="results_preexam")
            ]])
        )
        return
    
    emoji = "🏆" if result["score_negative"] >= 80 else "📊"
    
    text = f"""
{emoji} *{preexam['title']}*

✅ درست: {result['correct_answers']}
❌ غلط: {result['wrong_answers']}
⬜ بدون پاسخ: {result['blank_answers']}

📊 درصد ساده: {result['score_simple']:.2f}%
📉 درصد با نمره منفی: {result['score_negative']:.2f}%
"""
    
    keyboard = [[InlineKeyboardButton("🔙 بازگشت", callback_data="results_preexam")]]
    
    if result.get("answers_json"):
        keyboard.insert(0, [InlineKeyboardButton("📊 پاسخنامه کلیدی", callback_data=f"pre_analysis_{preexam_id}")])
    
    await query.message.edit_text(
        text,
        parse_mode="Markdown",
        reply_markup=InlineKeyboardMarkup(keyboard)
    )


async def back_to_results_menu(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """بازگشت به منوی نتایج."""
    query = update.callback_query
    await query.answer()
    
    keyboard = InlineKeyboardMarkup([
        [InlineKeyboardButton("📝 نتایج آزمون تمرینی", callback_data="results_exam")],
        [InlineKeyboardButton("📋 نتایج پیش‌آزمون", callback_data="results_preexam")],
        [InlineKeyboardButton("🔙 بازگشت", callback_data="back_to_menu")],
    ])
    
    await query.message.edit_text(
        "📊 *نتایج من*\n\n"
        "نتایج کدام آزمون را می‌خواهید مشاهده کنید؟",
        parse_mode="Markdown",
        reply_markup=keyboard
    )


async def show_support(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """نمایش پشتیبانی."""
    from config import SUPPORT_CHAT_ID
    
    keyboard = InlineKeyboardMarkup([
        [InlineKeyboardButton("💬 ورود به چت پشتیبانی", url=SUPPORT_CHAT_ID)],
    ])
    
    await update.message.reply_text(
        "📞 *پشتیبانی*\n\n"
        "برای ارتباط با پشتیبانی روی دکمه زیر کلیک کنید:",
        parse_mode="Markdown",
        reply_markup=keyboard
    )


async def show_about(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """نمایش معرفی مجموعه."""
    from repositories import SettingsRepository
    
    # دریافت اطلاعات پیام ذخیره شده
    about_chat_id = await SettingsRepository.get("about_chat_id")
    about_message_id = await SettingsRepository.get("about_message_id")
    
    if not about_chat_id or not about_message_id:
        await update.message.reply_text(
            "ℹ️ معرفی مجموعه هنوز تنظیم نشده است.\n\n"
            "🔜 به زودی..."
        )
        return
    
    try:
        # کپی کردن پیام بدون نمایش فرستنده اصلی
        await context.bot.copy_message(
            chat_id=update.effective_chat.id,
            from_chat_id=int(about_chat_id),
            message_id=int(about_message_id)
        )
    except Exception as e:
        await update.message.reply_text(
            "❌ خطا در نمایش معرفی مجموعه.\n"
            "لطفاً بعداً تلاش کنید."
        )


# ==================== هندلرهای تحلیل سوالات ====================

async def show_analysis(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """نمایش صفحه تحلیل سوالات با دکمه‌های ✅ و ❌."""
    query = update.callback_query
    await query.answer()
    
    answers = context.user_data.get("last_exam_answers")
    if not answers:
        await query.message.edit_text(
            "❌ اطلاعات آزمون یافت نشد. لطفاً آزمون جدید بدهید.",
            reply_markup=get_exam_result_keyboard()
        )
        return
    
    await query.message.edit_text(
        "📊 *تحلیل سوالات*\n\n"
        "لطفاً یکی از سوالات را برای مشاهده پاسخنامه تشریحی انتخاب کنید:",
        parse_mode="Markdown",
        reply_markup=get_analysis_keyboard(answers)
    )


async def show_question_review(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """نمایش پاسخنامه تشریحی یک سوال."""
    query = update.callback_query
    await query.answer()
    
    # استخراج شماره سوال از callback_data
    question_index = int(query.data.split("_")[1])
    
    answers = context.user_data.get("last_exam_answers")
    questions = context.user_data.get("last_exam_questions")
    
    if not answers or not questions or question_index >= len(answers):
        await query.answer("❌ خطا در دریافت اطلاعات", show_alert=True)
        return
    
    answer = answers[question_index]
    q = questions[question_index]
    # q = (id, text, option_a, option_b, option_c, option_d, correct_answer, explanation)
    
    option_labels = {"A": "الف", "B": "ب", "C": "ج", "D": "د"}
    
    # ساخت متن پاسخنامه
    user_answer_label = option_labels.get(answer["user_answer"], "نامشخص")
    correct_answer_label = option_labels.get(answer["correct_answer"], "نامشخص")
    
    status = "✅ *پاسخ صحیح*" if answer["is_correct"] else "❌ *پاسخ اشتباه*"
    
    text = f"""
📝 *سوال {question_index + 1}*

{q[1]}

*الف)* {q[2]}
*ب)* {q[3]}
*ج)* {q[4]}
*د)* {q[5]}

━━━━━━━━━━━━━━━
{status}

📌 *پاسخ شما:* {user_answer_label}
✔️ *پاسخ صحیح:* {correct_answer_label}
"""
    
    # اضافه کردن پاسخنامه تشریحی اگر وجود دارد
    if q[7]:  # explanation
        text += f"""
━━━━━━━━━━━━━━━
📖 *پاسخنامه تشریحی:*
{q[7]}
"""
    
    keyboard = [
        [InlineKeyboardButton("🔙 بازگشت به تحلیل", callback_data="show_analysis")],
        [InlineKeyboardButton("🏠 منوی اصلی", callback_data="back_to_menu")],
    ]
    
    await query.message.edit_text(
        text,
        parse_mode="Markdown",
        reply_markup=InlineKeyboardMarkup(keyboard)
    )


async def back_to_result(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """بازگشت به صفحه کارنامه."""
    query = update.callback_query
    await query.answer()
    
    result = context.user_data.get("last_exam_result")
    if not result:
        await query.message.edit_text(
            "❌ اطلاعات کارنامه یافت نشد.",
            reply_markup=get_exam_result_keyboard()
        )
        return
    
    await query.message.edit_text(
        format_result(result["correct"], result["wrong"], result["blank"], result["total"]),
        parse_mode="Markdown",
        reply_markup=get_exam_result_keyboard()
    )


# ==================== تابع دریافت هندلرها ====================

def get_exam_handlers() -> list:
    """برگرداندن لیست هندلرهای آزمون."""
    return [
        MessageHandler(filters.Regex("^📝 شروع آزمون$"), start_exam),
        MessageHandler(filters.Regex("^🎯 آزمون تراز \\| رایگان$"), show_taraz_menu),
        MessageHandler(filters.Regex("^👤 پروفایل$"), show_profile),
        MessageHandler(filters.Regex("^📊 نتایج من$"), show_results),
        MessageHandler(filters.Regex("^📞 پشتیبانی$"), show_support),
        MessageHandler(filters.Regex("^ℹ️ معرفی مجموعه$"), show_about),

        CallbackQueryHandler(handle_answer, pattern=r"^answer_\d+_[ABCD]$"),
        CallbackQueryHandler(back_to_menu, pattern="^back_to_menu$"),
        CallbackQueryHandler(new_exam_callback, pattern="^new_exam$"),
        CallbackQueryHandler(show_analysis, pattern="^show_analysis$"),
        CallbackQueryHandler(show_question_review, pattern=r"^review_\d+$"),
        CallbackQueryHandler(back_to_result, pattern="^back_to_result$"),
        # هندلرهای نتایج
        CallbackQueryHandler(show_exam_results, pattern="^results_exam$"),
        CallbackQueryHandler(show_preexam_results_menu, pattern="^results_preexam$"),
        CallbackQueryHandler(view_preexam_result_detail, pattern=r"^view_preexam_result_\d+$"),
        CallbackQueryHandler(back_to_results_menu, pattern="^back_to_results_menu$"),
    ]


async def show_taraz_menu(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """نمایش منوی آزمون تراز."""
    keyboard = InlineKeyboardMarkup([
        [InlineKeyboardButton("🎲 تست شانسی", callback_data="taraz_lucky")],
        [InlineKeyboardButton("🎨 طراحی تست", callback_data="taraz_design")],
        [InlineKeyboardButton("📋 پیش‌آزمون تراز", callback_data="taraz_preexam")],
        [InlineKeyboardButton("🔙 بازگشت", callback_data="back_to_menu")],
    ])
    
    await update.message.reply_text(
        "🎯 *آزمون تراز*\n\n"
        "یکی از گزینه‌ها را انتخاب کنید:",
        parse_mode="Markdown",
        reply_markup=keyboard
    )
