# handlers/preexam.py - هندلر پیش‌آزمون

import json
from telegram import (
    Update,
    InlineKeyboardButton,
    InlineKeyboardMarkup,
)
from telegram.ext import (
    ContextTypes,
    MessageHandler,
    CallbackQueryHandler,
    filters,
)

from config import NEGATIVE_SCORE_FACTOR
from repositories import PreexamRepository, PreexamResultRepository, UserRepository
from handlers.auth import get_main_menu_keyboard


# ==================== کیبوردها ====================

def get_preexam_question_keyboard(question_index: int) -> InlineKeyboardMarkup:
    """کیبورد گزینه‌های سوال پیش‌آزمون."""
    keyboard = [
        [
            InlineKeyboardButton("الف", callback_data=f"pre_ans_{question_index}_A"),
            InlineKeyboardButton("ب", callback_data=f"pre_ans_{question_index}_B"),
        ],
        [
            InlineKeyboardButton("ج", callback_data=f"pre_ans_{question_index}_C"),
            InlineKeyboardButton("د", callback_data=f"pre_ans_{question_index}_D"),
        ],
        [InlineKeyboardButton("⏭ رد کردن", callback_data=f"pre_skip_{question_index}")],
        [InlineKeyboardButton("🔙 انصراف", callback_data="pre_cancel")],
    ]
    return InlineKeyboardMarkup(keyboard)


# ==================== توابع کمکی ====================

def format_preexam_question(question: dict, current_num: int, total: int) -> str:
    """فرمت‌دهی سوال پیش‌آزمون."""
    return f"""
📝 *سوال {current_num} از {total}*

{question['text']}

*الف)* {question['option_a']}
*ب)* {question['option_b']}
*ج)* {question['option_c']}
*د)* {question['option_d']}
"""


def format_preexam_result(correct: int, wrong: int, blank: int, total: int, score_simple: float, score_negative: float) -> str:
    """فرمت‌دهی نتیجه پیش‌آزمون."""
    if score_negative >= 80:
        emoji = "🏆"
    elif score_negative >= 60:
        emoji = "👍"
    elif score_negative >= 40:
        emoji = "📚"
    else:
        emoji = "💪"
    
    return f"""
{emoji} *نتایج آزمون قبلی:*

✅ درست: {correct}
❌ غلط: {wrong}
⬜ بدون پاسخ: {blank}

📊 درصد ساده: {score_simple:.2f}%
📉 درصد با نمره منفی: {score_negative:.2f}%
"""


# ==================== هندلرها ====================

async def show_preexam_list(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """نمایش لیست پیش‌آزمون‌ها."""
    user_id = update.effective_user.id
    
    # چک ثبت‌نام
    db_user = await UserRepository.get_by_telegram_id(user_id)
    if not db_user or not db_user.is_registration_complete():
        await update.message.reply_text(
            "❌ لطفاً ابتدا ثبت‌نام را تکمیل کنید. دستور /start را ارسال کنید."
        )
        return
    
    # دریافت پیش‌آزمون‌های فعال
    preexams = await PreexamRepository.get_all_active()
    
    if not preexams:
        await update.message.reply_text(
            "❌ هیچ پیش‌آزمونی در حال حاضر موجود نیست.",
            reply_markup=get_main_menu_keyboard()
        )
        return
    
    # ساخت کیبورد
    keyboard = []
    for exam in preexams:
        # چک کردن نتیجه قبلی کاربر
        result = await PreexamResultRepository.get_user_result(user_id, exam["id"])
        
        if result:
            # کاربر قبلاً شرکت کرده - نمایش نتیجه
            emoji = "📊"
            text = f"{emoji} {exam['title']}"
        else:
            emoji = "📝"
            text = f"{emoji} {exam['title']}"
        
        keyboard.append([InlineKeyboardButton(text, callback_data=f"preexam_{exam['id']}")])
    
    # دکمه آرشیو
    keyboard.append([InlineKeyboardButton("🗂 آرشیو نتایج", callback_data="preexam_archive")])
    keyboard.append([InlineKeyboardButton("🔙 بازگشت", callback_data="back_to_menu")])
    
    await update.message.reply_text(
        "📋 *پیش‌آزمون تراز*\n\n"
        "آزمون مورد نظر خود را از منوی زیر انتخاب کنید:",
        parse_mode="Markdown",
        reply_markup=InlineKeyboardMarkup(keyboard)
    )


async def show_preexam_info(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """نمایش اطلاعات پیش‌آزمون."""
    query = update.callback_query
    await query.answer()
    
    preexam_id = int(query.data.split("_")[1])
    user_id = update.effective_user.id
    
    preexam = await PreexamRepository.get_by_id(preexam_id)
    if not preexam:
        await query.answer("❌ پیش‌آزمون یافت نشد", show_alert=True)
        return
    
    question_count = await PreexamRepository.get_question_count(preexam_id)
    result = await PreexamResultRepository.get_user_result(user_id, preexam_id)
    
    text = f"📝 *{preexam['title']}*\n\n"
    if preexam.get("description"):
        text += f"{preexam['description']}\n\n"
    text += f"📊 تعداد سوالات: {question_count}\n"
    
    keyboard = []
    
    if result:
        # نمایش نتیجه قبلی
        text += format_preexam_result(
            result["correct_answers"],
            result["wrong_answers"],
            result["blank_answers"],
            result["total_questions"],
            result["score_simple"],
            result["score_negative"]
        )
        
        # دکمه مشاهده تحلیل سوالات
        if result.get("answers_json"):
            keyboard.append([InlineKeyboardButton("📊 پاسخنامه کلیدی آزمون", callback_data=f"pre_analysis_{preexam_id}")])
        
        keyboard.append([InlineKeyboardButton("🔄 شرکت مجدد", callback_data=f"pre_start_{preexam_id}")])
    else:
        keyboard.append([InlineKeyboardButton("▶️ شروع آزمون", callback_data=f"pre_start_{preexam_id}")])
    
    keyboard.append([InlineKeyboardButton("🔙 بازگشت به لیست", callback_data="preexam_list")])
    
    await query.message.edit_text(
        text,
        parse_mode="Markdown",
        reply_markup=InlineKeyboardMarkup(keyboard)
    )


async def start_preexam(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """شروع پیش‌آزمون."""
    query = update.callback_query
    await query.answer()
    
    preexam_id = int(query.data.split("_")[2])
    user_id = update.effective_user.id
    
    preexam = await PreexamRepository.get_by_id(preexam_id)
    if not preexam:
        await query.answer("❌ پیش‌آزمون یافت نشد", show_alert=True)
        return
    
    questions = await PreexamRepository.get_questions(preexam_id)
    if not questions:
        await query.message.edit_text(
            "❌ این پیش‌آزمون هنوز سوالی ندارد.",
            reply_markup=InlineKeyboardMarkup([[
                InlineKeyboardButton("🔙 بازگشت", callback_data="preexam_list")
            ]])
        )
        return
    
    # ذخیره وضعیت آزمون
    context.user_data["preexam"] = {
        "preexam_id": preexam_id,
        "title": preexam["title"],
        "questions": questions,
        "current_index": 0,
        "answers": {},  # {index: {"answer": "A", "is_correct": True}}
    }
    
    # نمایش اولین سوال
    await show_preexam_question(query, context, 0)


async def show_preexam_question(query, context: ContextTypes.DEFAULT_TYPE, index: int) -> None:
    """نمایش سوال پیش‌آزمون."""
    exam = context.user_data.get("preexam")
    if not exam:
        return
    
    questions = exam["questions"]
    if index >= len(questions):
        return
    
    question = questions[index]
    
    await query.message.edit_text(
        format_preexam_question(question, index + 1, len(questions)),
        parse_mode="Markdown",
        reply_markup=get_preexam_question_keyboard(index)
    )


async def handle_preexam_answer(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """هندل کردن پاسخ پیش‌آزمون."""
    query = update.callback_query
    await query.answer()
    
    exam = context.user_data.get("preexam")
    if not exam:
        await query.message.edit_text("❌ آزمونی در جریان نیست.")
        return
    
    # پارس callback
    parts = query.data.split("_")
    action = parts[1]  # ans یا skip
    index = int(parts[2])
    
    questions = exam["questions"]
    question = questions[index]
    
    if action == "ans":
        user_answer = parts[3]
        is_correct = (user_answer == question["correct_answer"])
        exam["answers"][index] = {
            "answer": user_answer,
            "is_correct": is_correct,
            "correct_answer": question["correct_answer"],
            "question_text": question["text"][:50],
            "explanation": question.get("explanation", "")
        }
    else:  # skip
        exam["answers"][index] = {
            "answer": None,
            "is_correct": False,
            "correct_answer": question["correct_answer"],
            "question_text": question["text"][:50],
            "explanation": question.get("explanation", "")
        }
    
    # رفتن به سوال بعدی
    next_index = index + 1
    
    if next_index >= len(questions):
        # پایان آزمون
        await finish_preexam(query, context)
    else:
        exam["current_index"] = next_index
        context.user_data["preexam"] = exam
        await show_preexam_question(query, context, next_index)


async def finish_preexam(query, context: ContextTypes.DEFAULT_TYPE) -> None:
    """پایان پیش‌آزمون و ذخیره نتیجه."""
    exam = context.user_data.get("preexam")
    if not exam:
        return
    
    user_id = query.from_user.id
    preexam_id = exam["preexam_id"]
    questions = exam["questions"]
    answers = exam["answers"]
    
    # محاسبه نتایج
    total = len(questions)
    correct = sum(1 for a in answers.values() if a.get("is_correct"))
    wrong = sum(1 for a in answers.values() if a.get("answer") and not a.get("is_correct"))
    blank = total - correct - wrong
    
    score_simple = (correct / total) * 100 if total > 0 else 0
    effective_correct = correct - (wrong * NEGATIVE_SCORE_FACTOR)
    effective_correct = max(0, effective_correct)
    score_negative = (effective_correct / total) * 100 if total > 0 else 0
    
    # ذخیره در دیتابیس
    answers_json = json.dumps(answers, ensure_ascii=False)
    await PreexamResultRepository.save_result(
        user_id=user_id,
        preexam_id=preexam_id,
        total=total,
        correct=correct,
        wrong=wrong,
        blank=blank,
        score_simple=score_simple,
        score_negative=score_negative,
        answers_json=answers_json
    )
    
    # نمایش نتیجه
    text = f"🏆 *نتایج آزمون قبلی:*\n\n"
    text += f"✅ درست: {correct}\n"
    text += f"❌ غلط: {wrong}\n"
    text += f"⬜ بدون پاسخ: {blank}\n\n"
    text += f"📊 درصد ساده: {score_simple:.2f}%\n"
    text += f"📉 درصد با نمره منفی: {score_negative:.2f}%"
    
    keyboard = [
        [InlineKeyboardButton("📊 پاسخنامه کلیدی آزمون", callback_data=f"pre_analysis_{preexam_id}")],
        [InlineKeyboardButton("🔙 لیست پیش‌آزمون‌ها", callback_data="preexam_list")],
        [InlineKeyboardButton("🏠 منوی اصلی", callback_data="back_to_menu")],
    ]
    
    # پاک کردن وضعیت آزمون
    context.user_data.pop("preexam", None)
    
    await query.message.edit_text(
        text,
        parse_mode="Markdown",
        reply_markup=InlineKeyboardMarkup(keyboard)
    )


async def show_preexam_analysis(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """نمایش تحلیل سوالات پیش‌آزمون."""
    query = update.callback_query
    await query.answer()
    
    preexam_id = int(query.data.split("_")[2])
    user_id = update.effective_user.id
    
    result = await PreexamResultRepository.get_user_result(user_id, preexam_id)
    if not result or not result.get("answers_json"):
        await query.answer("❌ اطلاعات یافت نشد", show_alert=True)
        return
    
    answers = json.loads(result["answers_json"])
    
    # ذخیره برای استفاده بعدی
    context.user_data["preexam_analysis"] = {
        "preexam_id": preexam_id,
        "answers": answers
    }
    
    # ساخت کیبورد تحلیل
    keyboard = []
    for i in range(0, len(answers), 2):
        row = []
        
        # سوال اول
        ans1 = answers.get(str(i), {})
        emoji1 = "✅" if ans1.get("is_correct") else ("❌" if ans1.get("answer") else "⬜")
        row.append(InlineKeyboardButton(f"{emoji1} سوال {i + 1}", callback_data=f"pre_review_{i}"))
        
        # سوال دوم
        if str(i + 1) in answers:
            ans2 = answers.get(str(i + 1), {})
            emoji2 = "✅" if ans2.get("is_correct") else ("❌" if ans2.get("answer") else "⬜")
            row.append(InlineKeyboardButton(f"{emoji2} سوال {i + 2}", callback_data=f"pre_review_{i + 1}"))
        
        keyboard.append(row)
    
    keyboard.append([InlineKeyboardButton("🔙 بازگشت", callback_data=f"preexam_{preexam_id}")])
    
    await query.message.edit_text(
        "📊 *پاسخنامه کلیدی آزمون*\n\n"
        "لطفاً یکی از سوالات را برای تحلیل انتخاب کنید:",
        parse_mode="Markdown",
        reply_markup=InlineKeyboardMarkup(keyboard)
    )


async def show_preexam_question_review(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """نمایش بررسی سوال پیش‌آزمون."""
    query = update.callback_query
    await query.answer()
    
    question_index = int(query.data.split("_")[2])
    
    analysis = context.user_data.get("preexam_analysis")
    if not analysis:
        await query.answer("❌ اطلاعات یافت نشد", show_alert=True)
        return
    
    answers = analysis["answers"]
    preexam_id = analysis["preexam_id"]
    
    answer = answers.get(str(question_index), {})
    
    option_labels = {"A": "الف", "B": "ب", "C": "ج", "D": "د"}
    
    user_answer = option_labels.get(answer.get("answer"), "بدون پاسخ")
    correct_answer = option_labels.get(answer.get("correct_answer"), "نامشخص")
    
    if answer.get("is_correct"):
        status = "✅ *پاسخ صحیح*"
    elif answer.get("answer"):
        status = "❌ *پاسخ اشتباه*"
    else:
        status = "⬜ *بدون پاسخ*"
    
    text = f"📝 *سوال {question_index + 1}*\n\n"
    text += f"{answer.get('question_text', '')}...\n\n"
    text += f"━━━━━━━━━━━━━━━\n"
    text += f"{status}\n\n"
    text += f"📌 *پاسخ شما:* {user_answer}\n"
    text += f"✔️ *پاسخ صحیح:* {correct_answer}\n"
    
    if answer.get("explanation"):
        text += f"\n━━━━━━━━━━━━━━━\n"
        text += f"📖 *پاسخنامه تشریحی:*\n{answer['explanation']}"
    
    keyboard = [
        [InlineKeyboardButton("🔙 بازگشت به تحلیل", callback_data=f"pre_analysis_{preexam_id}")],
    ]
    
    await query.message.edit_text(
        text,
        parse_mode="Markdown",
        reply_markup=InlineKeyboardMarkup(keyboard)
    )


async def show_preexam_archive(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """نمایش آرشیو نتایج پیش‌آزمون."""
    query = update.callback_query
    await query.answer()
    
    user_id = update.effective_user.id
    results = await PreexamResultRepository.get_user_all_results(user_id)
    
    if not results:
        await query.message.edit_text(
            "🗂 *آرشیو نتایج*\n\n"
            "هنوز در هیچ پیش‌آزمونی شرکت نکرده‌اید.",
            parse_mode="Markdown",
            reply_markup=InlineKeyboardMarkup([[
                InlineKeyboardButton("🔙 بازگشت", callback_data="preexam_list")
            ]])
        )
        return
    
    text = "🗂 *آرشیو نتایج پیش‌آزمون‌ها:*\n\n"
    
    for result in results[:10]:  # آخرین 10 نتیجه
        emoji = "🏆" if result["score_negative"] >= 80 else "📊"
        text += f"{emoji} *{result['preexam_title']}*\n"
        text += f"   ✅{result['correct_answers']} ❌{result['wrong_answers']} ⬜{result['blank_answers']}\n"
        text += f"   📊 {result['score_negative']:.1f}%\n\n"
    
    keyboard = [[InlineKeyboardButton("🔙 بازگشت", callback_data="preexam_list")]]
    
    await query.message.edit_text(
        text,
        parse_mode="Markdown",
        reply_markup=InlineKeyboardMarkup(keyboard)
    )


async def cancel_preexam(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """انصراف از پیش‌آزمون."""
    query = update.callback_query
    await query.answer()
    
    context.user_data.pop("preexam", None)
    
    await query.message.edit_text(
        "❌ آزمون لغو شد.",
        reply_markup=InlineKeyboardMarkup([[
            InlineKeyboardButton("🔙 لیست پیش‌آزمون‌ها", callback_data="preexam_list")
        ]])
    )


async def go_to_preexam_list(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """بازگشت به لیست پیش‌آزمون‌ها."""
    query = update.callback_query
    await query.answer()
    
    user_id = update.effective_user.id
    preexams = await PreexamRepository.get_all_active()
    
    if not preexams:
        await query.message.edit_text(
            "❌ هیچ پیش‌آزمونی موجود نیست.",
            reply_markup=InlineKeyboardMarkup([[
                InlineKeyboardButton("🔙 منوی اصلی", callback_data="back_to_menu")
            ]])
        )
        return
    
    keyboard = []
    for exam in preexams:
        result = await PreexamResultRepository.get_user_result(user_id, exam["id"])
        emoji = "📊" if result else "📝"
        keyboard.append([InlineKeyboardButton(f"{emoji} {exam['title']}", callback_data=f"preexam_{exam['id']}")])
    
    keyboard.append([InlineKeyboardButton("🗂 آرشیو نتایج", callback_data="preexam_archive")])
    keyboard.append([InlineKeyboardButton("🔙 بازگشت", callback_data="back_to_menu")])
    
    await query.message.edit_text(
        "📋 *پیش‌آزمون تراز*\n\n"
        "آزمون مورد نظر خود را از منوی زیر انتخاب کنید:",
        parse_mode="Markdown",
        reply_markup=InlineKeyboardMarkup(keyboard)
    )


# ==================== تابع دریافت هندلرها ====================

def get_preexam_handlers() -> list:
    """برگرداندن لیست هندلرهای پیش‌آزمون."""
    return [
        MessageHandler(filters.Regex("^📋 پیش‌آزمون$"), show_preexam_list),
        CallbackQueryHandler(go_to_preexam_list, pattern="^preexam_list$"),
        CallbackQueryHandler(go_to_preexam_list, pattern="^taraz_preexam$"),
        CallbackQueryHandler(show_preexam_info, pattern=r"^preexam_\d+$"),
        CallbackQueryHandler(start_preexam, pattern=r"^pre_start_\d+$"),
        CallbackQueryHandler(handle_preexam_answer, pattern=r"^pre_(ans|skip)_\d+"),
        CallbackQueryHandler(show_preexam_analysis, pattern=r"^pre_analysis_\d+$"),
        CallbackQueryHandler(show_preexam_question_review, pattern=r"^pre_review_\d+$"),
        CallbackQueryHandler(show_preexam_archive, pattern="^preexam_archive$"),
        CallbackQueryHandler(cancel_preexam, pattern="^pre_cancel$"),
    ]
