# handlers/registration.py - ConversationHandler برای ثبت‌نام

from telegram import (
    Update,
    ReplyKeyboardMarkup,
    ReplyKeyboardRemove,
)
from telegram.ext import (
    ContextTypes,
    ConversationHandler,
    CommandHandler,
    MessageHandler,
    CallbackQueryHandler,
    filters,
)

from config import FIELDS, GRADES, PROVINCES
from services import ValidationService, TextService
from repositories import UserRepository
from handlers.auth import get_main_menu_keyboard


# ==================== State‌های ConversationHandler ====================
NAME, FIELD, GRADE, PROVINCE = range(4)


# ==================== کیبوردها ====================

def get_fields_keyboard() -> ReplyKeyboardMarkup:
    """کیبورد انتخاب رشته."""
    keyboard = [[field] for field in FIELDS]
    keyboard.append(["❌ انصراف"])
    return ReplyKeyboardMarkup(keyboard, resize_keyboard=True, one_time_keyboard=True)


def get_grades_keyboard() -> ReplyKeyboardMarkup:
    """کیبورد انتخاب پایه."""
    keyboard = [[grade] for grade in GRADES]
    keyboard.append(["⬅️ مرحله قبل", "❌ انصراف"])
    return ReplyKeyboardMarkup(keyboard, resize_keyboard=True, one_time_keyboard=True)


def get_provinces_keyboard() -> ReplyKeyboardMarkup:
    """کیبورد انتخاب استان."""
    # نمایش استان‌ها در ۲ ستون
    keyboard = []
    for i in range(0, len(PROVINCES), 2):
        row = [PROVINCES[i]]
        if i + 1 < len(PROVINCES):
            row.append(PROVINCES[i + 1])
        keyboard.append(row)
    keyboard.append(["⬅️ مرحله قبل", "❌ انصراف"])
    return ReplyKeyboardMarkup(keyboard, resize_keyboard=True, one_time_keyboard=True)


# ==================== هندلرها ====================

async def start_registration_callback(update: Update, context: ContextTypes.DEFAULT_TYPE) -> int:
    """شروع فرآیند ثبت‌نام از طریق دکمه inline (Entry Point)."""
    query = update.callback_query
    await query.answer()
    
    await query.message.edit_text(
        TextService.REGISTRATION_NAME,
        parse_mode="Markdown"
    )
    return NAME


async def receive_name(update: Update, context: ContextTypes.DEFAULT_TYPE) -> int:
    """دریافت نام کاربر."""
    name = update.message.text.strip()
    
    # اعتبارسنجی نام
    if not ValidationService.validate_name(name):
        await update.message.reply_text(
            "❌ نام وارد شده معتبر نیست. لطفاً نام و نام خانوادگی خود را به فارسی یا انگلیسی وارد کنید:"
        )
        return NAME
    
    # ذخیره موقت در user_data
    context.user_data["reg_name"] = name
    
    await update.message.reply_text(
        TextService.REGISTRATION_FIELD,
        parse_mode="Markdown",
        reply_markup=get_fields_keyboard()
    )
    return FIELD


async def receive_field(update: Update, context: ContextTypes.DEFAULT_TYPE) -> int:
    """دریافت رشته تحصیلی."""
    field = update.message.text.strip()
    
    if field == "❌ انصراف":
        return await cancel_registration(update, context)
    
    if field not in FIELDS:
        await update.message.reply_text(
            "❌ لطفاً یکی از گزینه‌های زیر را انتخاب کنید:",
            reply_markup=get_fields_keyboard()
        )
        return FIELD
    
    # ذخیره موقت
    context.user_data["reg_field"] = field
    
    await update.message.reply_text(
        TextService.REGISTRATION_GRADE,
        parse_mode="Markdown",
        reply_markup=get_grades_keyboard()
    )
    return GRADE


async def receive_grade(update: Update, context: ContextTypes.DEFAULT_TYPE) -> int:
    """دریافت پایه تحصیلی."""
    grade = update.message.text.strip()
    
    if grade == "❌ انصراف":
        return await cancel_registration(update, context)
    
    if grade == "⬅️ مرحله قبل":
        await update.message.reply_text(
            TextService.REGISTRATION_FIELD,
            parse_mode="Markdown",
            reply_markup=get_fields_keyboard()
        )
        return FIELD
    
    if grade not in GRADES:
        await update.message.reply_text(
            "❌ لطفاً یکی از گزینه‌های زیر را انتخاب کنید:",
            reply_markup=get_grades_keyboard()
        )
        return GRADE
    
    # ذخیره موقت
    context.user_data["reg_grade"] = grade
    
    await update.message.reply_text(
        TextService.REGISTRATION_PROVINCE,
        parse_mode="Markdown",
        reply_markup=get_provinces_keyboard()
    )
    return PROVINCE


async def receive_province(update: Update, context: ContextTypes.DEFAULT_TYPE) -> int:
    """دریافت استان و تکمیل ثبت‌نام."""
    province = update.message.text.strip()
    
    if province == "❌ انصراف":
        return await cancel_registration(update, context)
    
    if province == "⬅️ مرحله قبل":
        await update.message.reply_text(
            TextService.REGISTRATION_GRADE,
            parse_mode="Markdown",
            reply_markup=get_grades_keyboard()
        )
        return GRADE
    
    if province not in PROVINCES:
        await update.message.reply_text(
            "❌ لطفاً یکی از گزینه‌های زیر را انتخاب کنید:",
            reply_markup=get_provinces_keyboard()
        )
        return PROVINCE
    
    # ذخیره در دیتابیس
    user_id = update.effective_user.id
    await UserRepository.update_profile(
        telegram_id=user_id,
        full_name=context.user_data.get("reg_name"),
        field=context.user_data.get("reg_field"),
        grade=context.user_data.get("reg_grade"),
        province=province
    )
    
    # پاک کردن داده‌های موقت
    context.user_data.pop("reg_name", None)
    context.user_data.pop("reg_field", None)
    context.user_data.pop("reg_grade", None)
    
    # دریافت اطلاعات کاربر برای نمایش
    db_user = await UserRepository.get_by_telegram_id(user_id)
    
    # نمایش پیام تکمیل ثبت‌نام
    await update.message.reply_text(
        TextService.format_registration_complete(db_user),
        parse_mode="Markdown",
        reply_markup=ReplyKeyboardRemove()
    )
    
    # چک عضویت کانال بعد از ثبت‌نام
    from services import MembershipService
    from handlers.auth import get_join_channel_keyboard
    
    is_member = await MembershipService.check_channel_membership(
        context.bot, user_id
    )
    
    if not is_member:
        await update.message.reply_text(
            "👋 سلام! برای استفاده از همه امکانات ربات لازم است عضو کانال «تراز» باشید.\n\n"
            "1️⃣ اول به کانال ما بپیوندید.\n"
            "2️⃣ سپس دکمه «✅ عضو شدم» را بزنید تا دسترسی شما باز شود.",
            reply_markup=get_join_channel_keyboard()
        )
    else:
        # کاربر عضو است - نمایش منوی اصلی
        await update.message.reply_text(
            "🏠 *منوی اصلی*\n\nاز گزینه‌های زیر انتخاب کنید:",
            parse_mode="Markdown",
            reply_markup=get_main_menu_keyboard()
        )
    
    return ConversationHandler.END


async def cancel_registration(update: Update, context: ContextTypes.DEFAULT_TYPE) -> int:
    """انصراف از ثبت‌نام."""
    # پاک کردن داده‌های موقت
    context.user_data.pop("reg_name", None)
    context.user_data.pop("reg_field", None)
    context.user_data.pop("reg_grade", None)
    context.user_data.pop("awaiting_registration", None)
    
    await update.message.reply_text(
        "❌ ثبت‌نام لغو شد. برای شروع مجدد، دستور /start را ارسال کنید.",
        reply_markup=ReplyKeyboardRemove()
    )
    return ConversationHandler.END


# ==================== تابع دریافت هندلر ====================

def get_registration_handler() -> ConversationHandler:
    """برگرداندن ConversationHandler ثبت‌نام."""
    return ConversationHandler(
        entry_points=[
            CallbackQueryHandler(
                start_registration_callback,
                pattern="^start_registration$"
            ),
        ],
        states={
            NAME: [
                MessageHandler(filters.TEXT & ~filters.COMMAND, receive_name),
            ],
            FIELD: [
                MessageHandler(filters.TEXT & ~filters.COMMAND, receive_field),
            ],
            GRADE: [
                MessageHandler(filters.TEXT & ~filters.COMMAND, receive_grade),
            ],
            PROVINCE: [
                MessageHandler(filters.TEXT & ~filters.COMMAND, receive_province),
            ],
        },
        fallbacks=[
            CommandHandler("cancel", cancel_registration),
            MessageHandler(filters.Regex("^❌ انصراف$"), cancel_registration),
        ],
        allow_reentry=True,
    )
