# handlers/services.py - هندلر خدمات و محصولات

from telegram import (
    Update,
    InlineKeyboardButton,
    InlineKeyboardMarkup,
    ReplyKeyboardMarkup,
    ReplyKeyboardRemove,
)
from telegram.ext import (
    ContextTypes,
    ConversationHandler,
    MessageHandler,
    CallbackQueryHandler,
    filters,
)

from repositories import UserRepository, ServiceRequestRepository, SettingsRepository
from handlers.auth import get_main_menu_keyboard
from config import ADMIN_IDS


# State - افزوده شده state های جدید برای مشاوره
(
    WAITING_PHONE,
    WAITING_SELECTION,
    WAITING_RECEIPT,
    WAITING_PARENT_PHONE,
    WAITING_EXPECTATIONS,
    WAITING_CONSULT_RECEIPT,
) = range(6)


# ==================== کیبوردها ====================

def get_cancel_keyboard() -> ReplyKeyboardMarkup:
    """کیبورد انصراف."""
    return ReplyKeyboardMarkup(
        [["🔙 بازگشت"]],
        resize_keyboard=True,
        one_time_keyboard=True
    )


def get_services_keyboard() -> InlineKeyboardMarkup:
    """کیبورد خدمات."""
    keyboard = [
        [InlineKeyboardButton("📦 پکیج آزمون آزمایشی کنکور فرهنگیان", callback_data="service_package")],
        [InlineKeyboardButton("📅 رزرو مشاوره", callback_data="service_consult")],
        [InlineKeyboardButton("🔙 بازگشت", callback_data="back_to_menu")],
    ]
    return InlineKeyboardMarkup(keyboard)


def get_package_submenu_keyboard() -> InlineKeyboardMarkup:
    """کیبورد زیرمنوی پکیج."""
    keyboard = [
        [InlineKeyboardButton("📝 ثبت نام در آزمون", callback_data="pkg_register")],
        [InlineKeyboardButton("ℹ️ توضیحات و ویژگی‌ها", callback_data="pkg_info")],
        [InlineKeyboardButton("🔙 بازگشت", callback_data="back_to_services")],
    ]
    return InlineKeyboardMarkup(keyboard)


def get_consult_submenu_keyboard() -> InlineKeyboardMarkup:
    """کیبورد زیرمنوی مشاوره."""
    keyboard = [
        [InlineKeyboardButton("🔖 رزرو مشاوره تحصیلی", callback_data="consult_register")],
        [InlineKeyboardButton("ℹ️ توضیحات و ویژگی‌ها", callback_data="consult_info")],
        [InlineKeyboardButton("🔙 بازگشت", callback_data="back_to_services")],
    ]
    return InlineKeyboardMarkup(keyboard)


def get_parent_contact_keyboard() -> ReplyKeyboardMarkup:
    """کیبورد سوال والدین."""
    return ReplyKeyboardMarkup(
        [["مایل به ارتباط والدین نیستم."], ["بازگشت به منوی اصلی"]],
        resize_keyboard=True,
        one_time_keyboard=True
    )


def get_expectations_keyboard() -> ReplyKeyboardMarkup:
    """کیبورد انتظارات."""
    return ReplyKeyboardMarkup(
        [["انتظار خاص و مشاوری مدنظر ندارم."], ["بازگشت به منوی اصلی"]],
        resize_keyboard=True,
        one_time_keyboard=True
    )


# ==================== هندلرها ====================

async def start_services(update: Update, context: ContextTypes.DEFAULT_TYPE) -> int:
    """شروع بخش خدمات - دریافت شماره تماس."""
    user_id = update.effective_user.id
    
    # چک ثبت‌نام
    db_user = await UserRepository.get_by_telegram_id(user_id)
    if not db_user or not db_user.is_registration_complete():
        await update.message.reply_text(
            "❌ لطفاً ابتدا ثبت‌نام را تکمیل کنید. دستور /start را ارسال کنید."
        )
        return ConversationHandler.END
    
    await update.message.reply_text(
        "📞 *لطفاً شماره تماس صحیح و در دسترس خود را وارد کنید:*\n\n"
        "_(توجه کنید همین شماره ملاک ورود به پنل آزمون شما است)_",
        parse_mode="Markdown",
        reply_markup=get_cancel_keyboard()
    )
    return WAITING_PHONE


async def receive_phone(update: Update, context: ContextTypes.DEFAULT_TYPE) -> int:
    """دریافت شماره تماس و نمایش منوی خدمات."""
    text = update.message.text.strip()
    
    if text == "🔙 بازگشت":
        await update.message.reply_text(
            "🏠 منوی اصلی",
            reply_markup=get_main_menu_keyboard()
        )
        return ConversationHandler.END
    
    # اعتبارسنجی شماره
    phone = text.replace(" ", "").replace("-", "")
    if not (phone.startswith("09") and len(phone) == 11 and phone.isdigit()):
        await update.message.reply_text(
            "❌ شماره وارد شده معتبر نیست.\n"
            "لطفاً شماره موبایل صحیح وارد کنید (مثال: 09123456789):"
        )
        return WAITING_PHONE
    
    # ذخیره شماره در user_data
    context.user_data["service_phone"] = phone
    
    await update.message.reply_text(
        "🛍 *خدمات و محصولات موسسه آموزشی و مشاوره تراز*\n\n"
        "گزینه مورد نظر را انتخاب کنید:",
        parse_mode="Markdown",
        reply_markup=get_services_keyboard()
    )
    
    # حذف کیبورد reply
    await update.message.reply_text(
        "⬆️",
        reply_markup=ReplyKeyboardRemove()
    )
    
    return WAITING_SELECTION


async def handle_service_selection(update: Update, context: ContextTypes.DEFAULT_TYPE) -> int:
    """هندل انتخاب خدمات."""
    query = update.callback_query
    await query.answer()
    
    if query.data == "service_package":
        await query.message.edit_text(
            "📦 *پکیج آزمون آزمایشی کنکور فرهنگیان*\n\n"
            "لطفاً یکی از گزینه‌های زیر را انتخاب کنید:",
            parse_mode="Markdown",
            reply_markup=get_package_submenu_keyboard()
        )
        return WAITING_SELECTION

    elif query.data == "service_consult":
        await query.message.edit_text(
            "📅 *رزرو مشاوره تحصیلی*\n\n"
            "لطفاً یکی از گزینه‌های زیر را انتخاب کنید:",
            parse_mode="Markdown",
            reply_markup=get_consult_submenu_keyboard()
        )
        return WAITING_SELECTION

    elif query.data == "back_to_services":
        await query.message.edit_text(
            "🛍 *خدمات و محصولات موسسه آموزشی و مشاوره تراز*\n\n"
            "گزینه مورد نظر را انتخاب کنید:",
            parse_mode="Markdown",
            reply_markup=get_services_keyboard()
        )
        return WAITING_SELECTION

    elif query.data == "back_to_menu":
        await query.message.edit_text("🏠 منوی اصلی")
        return ConversationHandler.END
    
    return WAITING_SELECTION


async def handle_package_submenu(update: Update, context: ContextTypes.DEFAULT_TYPE) -> int:
    """هندل زیرمنوی پکیج."""
    query = update.callback_query
    user_id = update.effective_user.id
    
    if query.data == "pkg_info":
        await query.answer()
        await query.message.edit_text(
            "ℹ️ *توضیحات و ویژگی‌ها*\n\n"
            "برای مشاهده توضیحات کامل و ویژگی‌های آزمون، روی دکمه زیر کلیک کنید:",
            parse_mode="Markdown",
            reply_markup=InlineKeyboardMarkup([
                [InlineKeyboardButton("📋 مشاهده توضیحات کامل", url="https://t.me/TA_RAZ/37")],
                [InlineKeyboardButton("🔙 بازگشت", callback_data="service_package")]
            ])
        )
        return WAITING_SELECTION
        
    elif query.data == "pkg_register":
        # چک وجود درخواست قبلی در انتظار
        has_pending = await ServiceRequestRepository.has_pending_request(user_id)
        if has_pending:
            await query.answer(
                "⚠️ شما دارای رسید تعیین تکلیف نشده هستید. لطفا برای ثبت‌نام منتظر بمانید.",
                show_alert=True
            )
            return WAITING_SELECTION
        
        await query.answer()
        # نمایش اطلاعات پرداخت
        await query.message.delete()
        context.user_data["service_type"] = "پکیج آزمون الکترونیکی اختصاصی کنکور فرهنگیان تراز"
        
        # دریافت قیمت از دیتابیس
        exam_price = await SettingsRepository.get_exam_price()
        price_formatted = f"{exam_price:,}"
        
        payment_text = f"""پکیج آزمون الکترونیکی اختصاصی کنکور فرهنگیان تراز

💭 توضیحات: 

👈 برای خرید آزمون آزمایشی کنکور فرهنگیان لطفا مبلغ {price_formatted} تومان رو به شماره کارت زیر واریز کنید.

`6219861804771854`

🌱 به نام محمد حسام شهیدی

💳️️ بعد از واریز، اسکرین‌شات واریزی رو همینجا ارسال کنید."""
        await context.bot.send_message(
            chat_id=update.effective_chat.id,
            text=payment_text,
            parse_mode="Markdown",
            reply_markup=get_cancel_keyboard()
        )
        return WAITING_RECEIPT


    await query.answer()
    return WAITING_SELECTION


async def handle_consult_submenu(update: Update, context: ContextTypes.DEFAULT_TYPE) -> int:
    """هندل زیرمنوی مشاوره."""
    query = update.callback_query
    user_id = update.effective_user.id
    
    if query.data == "consult_info":
        await query.answer()
        await query.message.edit_text(
            "ℹ️ *توضیحات و ویژگی‌های مشاوره تحصیلی*\n\n"
            "برای مشاهده جزئیات کامل روی دکمه زیر کلیک کنید:",
            parse_mode="Markdown",
            reply_markup=InlineKeyboardMarkup([
                [InlineKeyboardButton("📖 مشاهده توضیحات", url="https://t.me/TA_RAZ/26")],
                [InlineKeyboardButton("🔙 بازگشت", callback_data="service_consult")]
            ])
        )
        return WAITING_SELECTION

        
    elif query.data == "consult_register":
        # چک وجود درخواست قبلی در انتظار
        has_pending = await ServiceRequestRepository.has_pending_request(user_id)
        if has_pending:
            await query.answer(
                "⚠️ شما دارای رسید تعیین تکلیف نشده هستید. لطفا برای ثبت‌نام منتظر بمانید.",
                show_alert=True
            )
            return WAITING_SELECTION
        
        await query.answer()
        # شروع فرآیند ثبت‌نام مشاوره - سوال اول: والدین
        await query.message.delete()
        
        await context.bot.send_message(
            chat_id=update.effective_chat.id,
            text="📅 *رزرو مشاوره*\n\n"
                 "✍️ آیا مایل هستید با والدین شما در ارتباط باشیم؟ در صورتی که مایل هستید شماره یکی از والدین خودتون رو ارسال کنید و در غیر این صورت از دکمه پایین صفحه استفاده کنید.",
            parse_mode="Markdown",
            reply_markup=get_parent_contact_keyboard()
        )
        return WAITING_PARENT_PHONE

    await query.answer()
    return WAITING_SELECTION


async def receive_parent_phone(update: Update, context: ContextTypes.DEFAULT_TYPE) -> int:
    """دریافت شماره والدین یا رد کردن."""
    text = update.message.text.strip()
    
    if text == "بازگشت به منوی اصلی":
        await update.message.reply_text(
            "🏠 منوی اصلی",
            reply_markup=get_main_menu_keyboard()
        )
        return ConversationHandler.END
    
    # ذخیره پاسخ
    if text == "مایل به ارتباط والدین نیستم.":
        context.user_data["parent_phone"] = None
    else:
        # فرض می‌کنیم شماره وارد شده
        context.user_data["parent_phone"] = text
    
    # مرحله دوم: انتظارات
    await update.message.reply_text(
        "✍️ وضعیت ثبت شد.\n\n"
        "📚 بهمون بگو که چه انتظاراتی از مشاورت داری؟ هر چقدر دوست داری برامون بنویس. با توجه به مواردی که مطرح می‌کنی ناظر به بهترین گزینه ممکن از بین تیم مشاوره وصلت می‌کنه. :)",
        reply_markup=get_expectations_keyboard()
    )
    return WAITING_EXPECTATIONS


async def receive_expectations(update: Update, context: ContextTypes.DEFAULT_TYPE) -> int:
    """دریافت انتظارات کاربر."""
    text = update.message.text.strip()
    
    if text == "بازگشت به منوی اصلی":
        await update.message.reply_text(
            "🏠 منوی اصلی",
            reply_markup=get_main_menu_keyboard()
        )
        return ConversationHandler.END
    
    # ذخیره انتظارات
    if text == "انتظار خاص و مشاوری مدنظر ندارم.":
        context.user_data["expectations"] = None
    else:
        context.user_data["expectations"] = text
    
    context.user_data["service_type"] = "consult"
    
    # دریافت قیمت مشاوره از دیتابیس
    consult_price = await SettingsRepository.get_consult_price()
    price_formatted = f"{consult_price:,}"
    
    # نمایش اطلاعات پرداخت مشاوره
    payment_text = f"""
🔖 رزرو [ ثبت نام ] مشاوره تحصیلی تراز 

📌 یک هفته فرصت عودت وجه 

💭 توضیحات: 

👈 برای ثبت نام 🔖 طرح «30 روزه مشاوره تحصیلی» لطفا مبلغ {price_formatted} تومان رو به شماره کارت زیر واریز کنید.

6219861804771854

🌱 به نام محمد حسام شهیدی

💳️️ بعد از واریز، اسکرین‌شات واریزی رو همینجا ارسال کنید.
"""
    await update.message.reply_text(
        payment_text,
        reply_markup=get_cancel_keyboard()
    )

    return WAITING_CONSULT_RECEIPT


async def receive_receipt(update: Update, context: ContextTypes.DEFAULT_TYPE) -> int:
    """دریافت فیش واریزی آزمون."""
    photo = update.message.photo[-1]
    file_id = photo.file_id
    
    user_id = update.effective_user.id
    phone = context.user_data.get("service_phone", "نامشخص")
    service_type = "پکیج آزمون (در انتظار تایید)"
    
    # ذخیره درخواست با فیش
    request_id = await ServiceRequestRepository.insert(user_id, phone, service_type, receipt_file_id=file_id)
    
    # دریافت نام
    db_user = await UserRepository.get_by_telegram_id(user_id)
    user_name = db_user.full_name if db_user else "ناشناس"
    
    await update.message.reply_text(
        "📌 رسید شما برای بررسی تیم مالی ارسال شد و بعد از تایید رسید، با شما ارتباط برقرار خواهد شد.\n\n"
        "- به منوی اصلی بازگشتید.",
        reply_markup=get_main_menu_keyboard()
    )
    
    # دریافت آیدی تلگرام
    tg_user = update.effective_user
    telegram_username = f"@{tg_user.username}" if tg_user.username else "آیدی ندارد"
    
    # اطلاع به ادمین
    await notify_admins(context, request_id, user_name, phone, service_type, telegram_username, file_id)
    
    return ConversationHandler.END


async def receive_consult_receipt(update: Update, context: ContextTypes.DEFAULT_TYPE) -> int:
    """دریافت فیش واریزی مشاوره."""
    photo = update.message.photo[-1]
    file_id = photo.file_id
    
    user_id = update.effective_user.id
    phone = context.user_data.get("service_phone", "نامشخص")
    parent_phone = context.user_data.get("parent_phone", "---")
    expectations = context.user_data.get("expectations", "---")
    service_type = "مشاوره تحصیلی (در انتظار تایید)"
    
    # ذخیره درخواست با فیش
    request_id = await ServiceRequestRepository.insert(user_id, phone, service_type, receipt_file_id=file_id)
    
    # دریافت نام
    db_user = await UserRepository.get_by_telegram_id(user_id)
    user_name = db_user.full_name if db_user else "ناشناس"
    
    await update.message.reply_text(
        "📌 رسید شما برای بررسی تیم مالی ارسال شد و بعد از تایید رسید، با شما ارتباط برقرار خواهد شد.\n\n"
        "- به منوی اصلی بازگشتید.",
        reply_markup=get_main_menu_keyboard()
    )
    
    # دریافت آیدی تلگرام
    tg_user = update.effective_user
    telegram_username = f"@{tg_user.username}" if tg_user.username else "آیدی ندارد"
    
    # اطلاع به ادمین با اطلاعات اضافی مشاوره
    await notify_admins_consult(context, request_id, user_name, phone, parent_phone, expectations, telegram_username, file_id)
    
    return ConversationHandler.END


async def notify_admins(context, request_id, user_name, phone, service_type, telegram_username, photo_id=None):
    """تابع کمکی ارسال اعلان به ادمین‌ها."""
    for admin_id in ADMIN_IDS:
        try:
            caption = (
                f"📥 *درخواست جدید (نیاز به بررسی)*\n\n"
                f"🆔 شناسه: {request_id}\n"
                f"👤 نام: {user_name}\n"
                f"🆔 آیدی تلگرام: {telegram_username}\n"
                f"📞 شماره: {phone}\n"
                f"📋 سرویس: {service_type}"
            )
            
            if photo_id:
                await context.bot.send_photo(
                    chat_id=admin_id,
                    photo=photo_id,
                    caption=caption,
                    parse_mode="Markdown"
                )
            else:
                await context.bot.send_message(
                    chat_id=admin_id,
                    text=caption,
                    parse_mode="Markdown"
                )
        except:
            pass



async def notify_admins_consult(context, request_id, user_name, phone, parent_phone, expectations, telegram_username, photo_id):
    """تابع کمکی ارسال اعلان مشاوره به ادمین‌ها."""
    for admin_id in ADMIN_IDS:
        try:
            caption = (
                f"📥 *درخواست مشاوره جدید (نیاز به بررسی)*\n\n"
                f"🆔 شناسه: {request_id}\n"
                f"👤 نام: {user_name}\n"
                f"🆔 آیدی تلگرام: {telegram_username}\n"
                f"📞 شماره: {phone}\n"
                f"👨‍👩‍👧 شماره والدین: {parent_phone if parent_phone else '---'}\n"
                f"📝 انتظارات: {expectations if expectations else '---'}\n"
                f"📋 سرویس: مشاوره تحصیلی"
            )
            
            await context.bot.send_photo(
                chat_id=admin_id,
                photo=photo_id,
                caption=caption,
                parse_mode="Markdown"
            )
        except:
            pass



async def cancel_services(update: Update, context: ContextTypes.DEFAULT_TYPE) -> int:
    """انصراف از خدمات."""
    await update.message.reply_text(
        "🏠 منوی اصلی",
        reply_markup=get_main_menu_keyboard()
    )
    return ConversationHandler.END


# ==================== تابع دریافت هندلرها ====================

def get_services_handlers() -> list:
    """برگرداندن لیست هندلرهای خدمات."""
    
    services_handler = ConversationHandler(
        entry_points=[
            MessageHandler(filters.Regex("^🛍 خدمات و محصولات$"), start_services),
        ],
        states={
            WAITING_PHONE: [
                MessageHandler(filters.TEXT & ~filters.COMMAND, receive_phone),
            ],
            WAITING_SELECTION: [
                CallbackQueryHandler(handle_service_selection, pattern="^service_(package|consult)$"),
                CallbackQueryHandler(handle_package_submenu, pattern="^pkg_(register|info)$"),
                CallbackQueryHandler(handle_consult_submenu, pattern="^consult_(register|info)$"),
                CallbackQueryHandler(handle_service_selection, pattern="^back_to_services$"),
                CallbackQueryHandler(handle_service_selection, pattern="^back_to_menu$"),
            ],
            WAITING_RECEIPT: [
                MessageHandler(filters.PHOTO, receive_receipt),
                MessageHandler(filters.Regex("^🔙 بازگشت$"), cancel_services),
            ],
            WAITING_PARENT_PHONE: [
                MessageHandler(filters.TEXT & ~filters.COMMAND, receive_parent_phone),
            ],
            WAITING_EXPECTATIONS: [
                MessageHandler(filters.TEXT & ~filters.COMMAND, receive_expectations),
            ],
            WAITING_CONSULT_RECEIPT: [
                MessageHandler(filters.PHOTO, receive_consult_receipt),
                MessageHandler(filters.Regex("^🔙 بازگشت$"), cancel_services),
            ],
        },
        fallbacks=[
            MessageHandler(filters.Regex("^🔙 بازگشت$"), cancel_services),
            MessageHandler(filters.Regex("^بازگشت به منوی اصلی$"), cancel_services),
        ],
        allow_reentry=True,
    )
    
    return [services_handler]
