# handlers/user_design.py - هندلر طراحی تست توسط کاربران و تست شانسی

from telegram import (
    Update,
    InlineKeyboardButton,
    InlineKeyboardMarkup,
    ReplyKeyboardMarkup,
    ReplyKeyboardRemove,
)
from telegram.ext import (
    ContextTypes,
    ConversationHandler,
    CommandHandler,
    MessageHandler,
    CallbackQueryHandler,
    filters,
)

from repositories import PendingQuestionRepository, UserQuestionRepository, UserRepository
from handlers.auth import get_main_menu_keyboard


# ==================== State‌های ConversationHandler ====================
(
    DESIGN_TEXT,
    DESIGN_OPTION_A,
    DESIGN_OPTION_B,
    DESIGN_OPTION_C,
    DESIGN_OPTION_D,
    DESIGN_CORRECT,
    DESIGN_EXPLANATION,
) = range(7)


# ==================== کیبوردها ====================

def get_cancel_keyboard() -> ReplyKeyboardMarkup:
    """کیبورد انصراف."""
    return ReplyKeyboardMarkup(
        [["❌ انصراف"]],
        resize_keyboard=True,
        one_time_keyboard=True
    )


def get_correct_answer_keyboard() -> InlineKeyboardMarkup:
    """کیبورد انتخاب گزینه صحیح."""
    keyboard = [
        [
            InlineKeyboardButton("الف (A)", callback_data="design_correct_A"),
            InlineKeyboardButton("ب (B)", callback_data="design_correct_B"),
        ],
        [
            InlineKeyboardButton("ج (C)", callback_data="design_correct_C"),
            InlineKeyboardButton("د (D)", callback_data="design_correct_D"),
        ],
        [InlineKeyboardButton("❌ انصراف", callback_data="design_cancel")],
    ]
    return InlineKeyboardMarkup(keyboard)


def get_skip_keyboard() -> ReplyKeyboardMarkup:
    """کیبورد رد کردن."""
    return ReplyKeyboardMarkup(
        [["⏭ رد کردن"], ["❌ انصراف"]],
        resize_keyboard=True,
        one_time_keyboard=True
    )


def get_lucky_test_keyboard() -> InlineKeyboardMarkup:
    """کیبورد تست شانسی."""
    keyboard = [
        [
            InlineKeyboardButton("الف", callback_data="lucky_A"),
            InlineKeyboardButton("ب", callback_data="lucky_B"),
        ],
        [
            InlineKeyboardButton("ج", callback_data="lucky_C"),
            InlineKeyboardButton("د", callback_data="lucky_D"),
        ],
        [InlineKeyboardButton("🔙 بازگشت به منو", callback_data="back_to_menu")],
    ]
    return InlineKeyboardMarkup(keyboard)


# ==================== هندلرهای طراحی تست ====================

async def start_design_test(update: Update, context: ContextTypes.DEFAULT_TYPE) -> int:
    """شروع فرآیند طراحی تست."""
    user_id = update.effective_user.id
    
    # چک ثبت‌نام
    db_user = await UserRepository.get_by_telegram_id(user_id)
    if not db_user or not db_user.is_registration_complete():
        await update.message.reply_text(
            "❌ لطفاً ابتدا ثبت‌نام را تکمیل کنید. دستور /start را ارسال کنید."
        )
        return ConversationHandler.END
    
    await update.message.reply_text(
        "🧩 *به بخش طراحی تست خوش آمدید!* 🌟\n\n"
        "طراحی تست یکی از بهترین روش‌های یادگیری است. وقتی خودتان سوالات تستی می‌سازید، "
        "مطالب عمیق‌تر در ذهن‌تان حک می‌شود و به خاطر سپردن آن‌ها آسان‌تر می‌گردد.\n\n"
        "علاوه بر این، تست‌های شما در دسترس سایر کاربران قرار می‌گیرد، "
        "آن‌ها به تست‌هایتان پاسخ می‌دهند و بر اساس کیفیت و جذابیت، به شما امتیاز می‌دهند. "
        "این امتیازها به شما انگیزه می‌دهد تا تست‌های بهتری بسازید و جامعه ربات را غنی‌تر کنید!\n\n"
        "💡 *نکته:* تست‌هایتان پس از بررسی توسط ادمین، تایید می‌شوند و "
        "در بخش «تست شانسی» برای دیگران نمایش داده می‌شوند. "
        "هر تست تاییدشده می‌تواند هزاران بار زده شود و امتیاز جمع کند.\n\n"
        "━━━━━━━━━━━━━━━\n"
        "📝 *مرحله ۱ از ۷*\n"
        "لطفاً متن سوال را وارد کنید:",
        parse_mode="Markdown",
        reply_markup=get_cancel_keyboard()
    )
    return DESIGN_TEXT


async def receive_design_text(update: Update, context: ContextTypes.DEFAULT_TYPE) -> int:
    """دریافت متن سوال."""
    text = update.message.text.strip()
    
    if text == "❌ انصراف":
        return await cancel_design(update, context)
    
    if len(text) < 10:
        await update.message.reply_text(
            "❌ متن سوال باید حداقل ۱۰ کاراکتر باشد. دوباره تلاش کنید:"
        )
        return DESIGN_TEXT
    
    context.user_data["design_question"] = {"text": text}
    
    await update.message.reply_text(
        "📝 *مرحله ۲ از ۷*\n\nگزینه الف را وارد کنید:",
        parse_mode="Markdown",
        reply_markup=get_cancel_keyboard()
    )
    return DESIGN_OPTION_A


async def receive_design_option_a(update: Update, context: ContextTypes.DEFAULT_TYPE) -> int:
    """دریافت گزینه الف."""
    text = update.message.text.strip()
    
    if text == "❌ انصراف":
        return await cancel_design(update, context)
    
    context.user_data["design_question"]["option_a"] = text
    
    await update.message.reply_text(
        "📝 *مرحله ۳ از ۷*\n\nگزینه ب را وارد کنید:",
        parse_mode="Markdown",
        reply_markup=get_cancel_keyboard()
    )
    return DESIGN_OPTION_B


async def receive_design_option_b(update: Update, context: ContextTypes.DEFAULT_TYPE) -> int:
    """دریافت گزینه ب."""
    text = update.message.text.strip()
    
    if text == "❌ انصراف":
        return await cancel_design(update, context)
    
    context.user_data["design_question"]["option_b"] = text
    
    await update.message.reply_text(
        "📝 *مرحله ۴ از ۷*\n\nگزینه ج را وارد کنید:",
        parse_mode="Markdown",
        reply_markup=get_cancel_keyboard()
    )
    return DESIGN_OPTION_C


async def receive_design_option_c(update: Update, context: ContextTypes.DEFAULT_TYPE) -> int:
    """دریافت گزینه ج."""
    text = update.message.text.strip()
    
    if text == "❌ انصراف":
        return await cancel_design(update, context)
    
    context.user_data["design_question"]["option_c"] = text
    
    await update.message.reply_text(
        "📝 *مرحله ۵ از ۷*\n\nگزینه د را وارد کنید:",
        parse_mode="Markdown",
        reply_markup=get_cancel_keyboard()
    )
    return DESIGN_OPTION_D


async def receive_design_option_d(update: Update, context: ContextTypes.DEFAULT_TYPE) -> int:
    """دریافت گزینه د."""
    text = update.message.text.strip()
    
    if text == "❌ انصراف":
        return await cancel_design(update, context)
    
    context.user_data["design_question"]["option_d"] = text
    
    q = context.user_data["design_question"]
    summary = (
        f"📝 *مرحله ۶ از ۷ - انتخاب گزینه صحیح*\n\n"
        f"*سوال:*\n{q['text']}\n\n"
        f"*الف)* {q['option_a']}\n"
        f"*ب)* {q['option_b']}\n"
        f"*ج)* {q['option_c']}\n"
        f"*د)* {q['option_d']}\n\n"
        f"گزینه صحیح را انتخاب کنید:"
    )
    
    await update.message.reply_text(
        summary,
        parse_mode="Markdown",
        reply_markup=get_correct_answer_keyboard()
    )
    return DESIGN_CORRECT


async def receive_design_correct(update: Update, context: ContextTypes.DEFAULT_TYPE) -> int:
    """دریافت گزینه صحیح."""
    query = update.callback_query
    await query.answer()
    
    if query.data == "design_cancel":
        return await cancel_design_callback(update, context)
    
    correct = query.data.split("_")[2]  # design_correct_A -> A
    context.user_data["design_question"]["correct_answer"] = correct
    
    await query.message.edit_text(
        "📝 *مرحله ۷ از ۷ - پاسخنامه تشریحی*\n\n"
        "توضیح تشریحی پاسخ را وارد کنید:\n"
        "(می‌توانید با «⏭ رد کردن» این مرحله را رد کنید)",
        parse_mode="Markdown"
    )
    
    await query.message.reply_text(
        "پاسخنامه تشریحی:",
        reply_markup=get_skip_keyboard()
    )
    return DESIGN_EXPLANATION


async def receive_design_explanation(update: Update, context: ContextTypes.DEFAULT_TYPE) -> int:
    """دریافت پاسخنامه و ارسال به ادمین."""
    text = update.message.text.strip()
    
    if text == "❌ انصراف":
        return await cancel_design(update, context)
    
    explanation = None if text == "⏭ رد کردن" else text
    
    q = context.user_data["design_question"]
    user_id = update.effective_user.id
    
    # ذخیره در دیتابیس
    question_id = await PendingQuestionRepository.insert(
        text=q["text"],
        option_a=q["option_a"],
        option_b=q["option_b"],
        option_c=q["option_c"],
        option_d=q["option_d"],
        correct_answer=q["correct_answer"],
        explanation=explanation,
        submitted_by=user_id
    )
    
    # پاک کردن داده موقت
    context.user_data.pop("design_question", None)
    
    correct_labels = {"A": "الف", "B": "ب", "C": "ج", "D": "د"}
    
    await update.message.reply_text(
        f"✅ *تست شما ارسال شد!*\n\n"
        f"🆔 شناسه: {question_id}\n"
        f"✔️ گزینه صحیح: {correct_labels[q['correct_answer']]}\n\n"
        f"تست شما پس از بررسی و تایید ادمین در مخزن تست شانسی قرار می‌گیرد.\n"
        f"با تشکر از مشارکت شما! 🎉",
        parse_mode="Markdown",
        reply_markup=get_main_menu_keyboard()
    )
    
    # اطلاع به ادمین‌ها
    from config import ADMIN_IDS
    for admin_id in ADMIN_IDS:
        try:
            db_user = await UserRepository.get_by_telegram_id(user_id)
            user_name = db_user.full_name if db_user else "ناشناس"
            
            await context.bot.send_message(
                chat_id=admin_id,
                text=f"📬 *تست جدید ارسال شد!*\n\n"
                     f"👤 ارسال‌کننده: {user_name}\n"
                     f"🆔 شناسه: {question_id}\n\n"
                     f"برای بررسی از /admin استفاده کنید.",
                parse_mode="Markdown"
            )
        except:
            pass
    
    return ConversationHandler.END


async def cancel_design(update: Update, context: ContextTypes.DEFAULT_TYPE) -> int:
    """انصراف از طراحی تست."""
    context.user_data.pop("design_question", None)
    
    await update.message.reply_text(
        "❌ طراحی تست لغو شد.",
        reply_markup=get_main_menu_keyboard()
    )
    return ConversationHandler.END


async def cancel_design_callback(update: Update, context: ContextTypes.DEFAULT_TYPE) -> int:
    """انصراف از طراحی تست (از کالبک)."""
    query = update.callback_query
    await query.answer()
    
    context.user_data.pop("design_question", None)
    
    await query.message.edit_text("❌ طراحی تست لغو شد.")
    await query.message.reply_text(
        "🏠 منوی اصلی",
        reply_markup=get_main_menu_keyboard()
    )
    return ConversationHandler.END


# ==================== هندلرهای تست شانسی ====================

async def start_lucky_test(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """شروع تست شانسی."""
    user_id = update.effective_user.id
    
    # چک ثبت‌نام
    db_user = await UserRepository.get_by_telegram_id(user_id)
    if not db_user or not db_user.is_registration_complete():
        await update.message.reply_text(
            "❌ لطفاً ابتدا ثبت‌نام را تکمیل کنید. دستور /start را ارسال کنید."
        )
        return
    
    # دریافت سوال تصادفی از مخزن B (سوالاتی که کاربر قبلا دیده exclude می‌شوند)
    question = await UserQuestionRepository.get_random_one(user_id=user_id)
    
    if not question:
        await update.message.reply_text(
            "❌ تست‌ها تموم شدن! حالا برو بخش طراحی تست و تست‌های خودت رو بساز تا بقیه هم حل کنن و امتیاز بدن. راستی، همین که خودت تست درست می‌کنی، کمک می‌کنه مطالب بهتر تو ذهنت بمونه 📚",
            reply_markup=get_main_menu_keyboard()
        )
        return
    
    # ثبت اینکه کاربر این سوال را دیده (جلوگیری از تکرار)
    await UserQuestionRepository.mark_seen(user_id, question['id'])
    
    # ذخیره سوال فعلی در user_data
    context.user_data["lucky_question"] = question
    
    await update.message.reply_text(
        f"🎲 *تست شانسی*\n\n"
        f"{question['text']}\n\n"
        f"*الف)* {question['option_a']}\n"
        f"*ب)* {question['option_b']}\n"
        f"*ج)* {question['option_c']}\n"
        f"*د)* {question['option_d']}",
        parse_mode="Markdown",
        reply_markup=get_lucky_test_keyboard()
    )


async def handle_lucky_answer(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """هندل کردن پاسخ تست شانسی."""
    query = update.callback_query
    await query.answer()
    
    user_answer = query.data.split("_")[1]  # lucky_A -> A
    
    question = context.user_data.get("lucky_question")
    if not question:
        await query.message.edit_text(
            "❌ خطا در دریافت سوال. لطفاً دوباره تلاش کنید.",
            reply_markup=InlineKeyboardMarkup([[
                InlineKeyboardButton("🔙 بازگشت به منو", callback_data="back_to_menu")
            ]])
        )
        return
    
    option_labels = {"A": "الف", "B": "ب", "C": "ج", "D": "د"}
    correct_answer = question["correct_answer"]
    is_correct = (user_answer == correct_answer)
    
    # ساخت متن نتیجه
    maker_name = question.get('maker_name', 'ناشناس')
    
    if is_correct:
        result = "✅ *درست!*"
    else:
        result = f"❌ *اشتباه!*\n\n📌 پاسخ شما: {option_labels[user_answer]}\n✔️ پاسخ صحیح: {option_labels[correct_answer]}"
    
    text = f"""
🎲 *نتیجه تست شانسی*
👤 طراح: {maker_name}

{question['text']}

*الف)* {question['option_a']}
*ب)* {question['option_b']}
*ج)* {question['option_c']}
*د)* {question['option_d']}

━━━━━━━━━━━━━━━
{result}
"""
    
    # اضافه کردن پاسخنامه تشریحی
    if question.get("explanation"):
        text += f"""
━━━━━━━━━━━━━━━
📖 *پاسخنامه تشریحی:*
{question['explanation']}
"""

    text += "\n⭐️ *به این سوال امتیاز دهید:*"
    
    # کیبورد امتیازدهی
    rating_row = [
        InlineKeyboardButton("⭐️ 1", callback_data=f"rate_{question['id']}_1"),
        InlineKeyboardButton("⭐️ 2", callback_data=f"rate_{question['id']}_2"),
        InlineKeyboardButton("⭐️ 3", callback_data=f"rate_{question['id']}_3"),
        InlineKeyboardButton("⭐️ 4", callback_data=f"rate_{question['id']}_4"),
        InlineKeyboardButton("⭐️ 5", callback_data=f"rate_{question['id']}_5"),
    ]
    
    keyboard = [
        rating_row,
        [InlineKeyboardButton("🎲 تست شانسی دیگر", callback_data="another_lucky")],
        [InlineKeyboardButton("🏠 منوی اصلی", callback_data="back_to_menu")],
    ]
    
    await query.message.edit_text(
        text,
        parse_mode="Markdown",
        reply_markup=InlineKeyboardMarkup(keyboard)
    )


async def handle_rating(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """ثبت امتیاز کاربر."""
    query = update.callback_query
    # await query.answer() # We will answer with specific message
    
    data = query.data.split("_")
    question_id = int(data[1])
    score = int(data[2])
    user_id = update.effective_user.id
    
    from repositories import RatingRepository
    
    success = await RatingRepository.add_rating(user_id, question_id, score)
    
    if success:
        await query.answer(f"✅ امتیاز {score} ثبت شد. ممنون!", show_alert=False)
        
        # آپدیت کیبورد برای حذف دکمه‌های امتیازدهی (چون فقط یکبار میشه رای داد)
        current_markup = query.message.reply_markup
        new_keyboard = []
        for row in current_markup.inline_keyboard:
            # دکمه‌های غیر از امتیازدهی را نگه می‌داریم
            # فرض می‌کنیم دکمه‌های امتیازدهی با callback_data="rate_..." شروع می‌شوند
            if not row[0].callback_data.startswith("rate"):
                new_keyboard.append(row)
            else:
                # جایگزینی ردیف امتیاز با پیام
                new_keyboard.append([InlineKeyboardButton(f"✅ امتیاز شما: {score} ستاره", callback_data="noop")])
        
        await query.message.edit_reply_markup(reply_markup=InlineKeyboardMarkup(new_keyboard))
    else:
        await query.answer("❌ شما قبلاً به این سوال رای داده‌اید!", show_alert=True)


async def another_lucky_test(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """تست شانسی دیگر."""
    query = update.callback_query
    await query.answer()
    
    # دریافت سوال تصادفی جدید
    question = await UserQuestionRepository.get_random_one()
    
    if not question:
        await query.message.edit_text(
            "❌ متاسفانه سوال دیگری در مخزن موجود نیست.",
            reply_markup=InlineKeyboardMarkup([[
                InlineKeyboardButton("🏠 منوی اصلی", callback_data="back_to_menu")
            ]])
        )
        return
    
    context.user_data["lucky_question"] = question
    
    await query.message.edit_text(
        f"🎲 *تست شانسی*\n\n"
        f"{question['text']}\n\n"
        f"*الف)* {question['option_a']}\n"
        f"*ب)* {question['option_b']}\n"
        f"*ج)* {question['option_c']}\n"
        f"*د)* {question['option_d']}",
        parse_mode="Markdown",
        reply_markup=get_lucky_test_keyboard()
    )


# ==================== تابع دریافت هندلرها ====================

def get_user_design_handlers() -> list:
    """برگرداندن لیست هندلرهای طراحی تست و تست شانسی."""
    
    # ConversationHandler برای طراحی تست
    design_handler = ConversationHandler(
        entry_points=[
            MessageHandler(filters.Regex("^🎨 طراحی تست$"), start_design_test),
            CallbackQueryHandler(start_create_test, pattern="^create_test$"),
        ],
        states={
            DESIGN_TEXT: [
                MessageHandler(filters.TEXT & ~filters.COMMAND, receive_design_text),
            ],
            DESIGN_OPTION_A: [
                MessageHandler(filters.TEXT & ~filters.COMMAND, receive_design_option_a),
            ],
            DESIGN_OPTION_B: [
                MessageHandler(filters.TEXT & ~filters.COMMAND, receive_design_option_b),
            ],
            DESIGN_OPTION_C: [
                MessageHandler(filters.TEXT & ~filters.COMMAND, receive_design_option_c),
            ],
            DESIGN_OPTION_D: [
                MessageHandler(filters.TEXT & ~filters.COMMAND, receive_design_option_d),
            ],
            DESIGN_CORRECT: [
                CallbackQueryHandler(receive_design_correct, pattern="^design_(correct_[ABCD]|cancel)$"),
            ],
            DESIGN_EXPLANATION: [
                MessageHandler(filters.TEXT & ~filters.COMMAND, receive_design_explanation),
            ],
        },
        fallbacks=[
            MessageHandler(filters.Regex("^❌ انصراف$"), cancel_design),
            CallbackQueryHandler(cancel_design_callback, pattern="^design_cancel$"),
        ],
        allow_reentry=True,
    )
    
    return [
        design_handler,
        MessageHandler(filters.Regex("^🎲 تست شانسی$"), start_lucky_test),
        CallbackQueryHandler(start_design_test_callback, pattern="^taraz_design$"),
        CallbackQueryHandler(show_my_tests, pattern="^my_tests$"),
        CallbackQueryHandler(start_lucky_test_callback, pattern="^taraz_lucky$"),
        CallbackQueryHandler(handle_lucky_answer, pattern=r"^lucky_[ABCD]$"),
        CallbackQueryHandler(handle_rating, pattern=r"^rate_\d+_[1-5]$"),
        CallbackQueryHandler(another_lucky_test, pattern="^another_lucky$"),
    ]


async def start_design_test_callback(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """نمایش منوی طراحی تست."""
    query = update.callback_query
    await query.answer()
    
    user_id = update.effective_user.id
    
    db_user = await UserRepository.get_by_telegram_id(user_id)
    if not db_user or not db_user.is_registration_complete():
        await query.message.edit_text(
            "❌ لطفاً ابتدا ثبت‌نام را تکمیل کنید. دستور /start را ارسال کنید."
        )
        return
    
    keyboard = InlineKeyboardMarkup([
        [InlineKeyboardButton("📋 تست‌های من", callback_data="my_tests")],
        [InlineKeyboardButton("➕ ایجاد تست", callback_data="create_test")],
        [InlineKeyboardButton("🔙 بازگشت", callback_data="back_to_menu")],
    ])
    
    await query.message.edit_text(
        "🧩 *به بخش طراحی تست خوش آمدید!* 🌟\n\n"
        "طراحی تست یکی از بهترین روش‌های یادگیری است. وقتی خودتان سوالات تستی می‌سازید، "
        "مطالب عمیق‌تر در ذهن‌تان حک می‌شود و به خاطر سپردن آن‌ها آسان‌تر می‌گردد.\n\n"
        "علاوه بر این، تست‌های شما در دسترس سایر کاربران قرار می‌گیرد، "
        "آن‌ها به تست‌هایتان پاسخ می‌دهند و بر اساس کیفیت و جذابیت، به شما امتیاز می‌دهند. "
        "این امتیازها به شما انگیزه می‌دهد تا تست‌های بهتری بسازید و جامعه ربات را غنی‌تر کنید!\n\n"
        "💡 *نکته:* تست‌هایتان پس از بررسی توسط ادمین، تایید می‌شوند و "
        "در بخش «تست شانسی» برای دیگران نمایش داده می‌شوند. "
        "هر تست تاییدشده می‌تواند هزاران بار زده شود و امتیاز جمع کند.\n\n"
        "گزینه مورد نظر را انتخاب کنید:",
        parse_mode="Markdown",
        reply_markup=keyboard
    )


async def start_create_test(update: Update, context: ContextTypes.DEFAULT_TYPE) -> int:
    """شروع فرآیند ایجاد تست."""
    query = update.callback_query
    await query.answer()
    
    await query.message.edit_text(
        "📝 *ایجاد تست جدید*\n\n"
        "*مرحله ۱ از ۷*\n"
        "لطفاً متن سوال را وارد کنید:",
        parse_mode="Markdown"
    )
    
    await query.message.reply_text(
        "متن سوال را وارد کنید:",
        reply_markup=get_cancel_keyboard()
    )
    return DESIGN_TEXT


async def show_my_tests(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """نمایش تست‌های کاربر."""
    query = update.callback_query
    await query.answer()
    
    user_id = update.effective_user.id
    
    # دریافت تست‌های تایید شده کاربر
    tests = await UserQuestionRepository.get_by_user(user_id)
    
    # دریافت تست‌های در انتظار کاربر
    pending = await PendingQuestionRepository.get_by_user(user_id)
    
    text = "📋 *تست‌های من*\n\n"
    
    if tests:
        text += f"✅ *تست‌های تایید شده:* {len(tests)}\n"
    else:
        text += "✅ هنوز تستی تایید نشده است.\n"
    
    if pending:
        text += f"⏳ *در انتظار تایید:* {len(pending)}\n"
    else:
        text += "⏳ تستی در انتظار تایید نیست.\n"
    
    keyboard = InlineKeyboardMarkup([
        [InlineKeyboardButton("➕ ایجاد تست جدید", callback_data="create_test")],
        [InlineKeyboardButton("🔙 بازگشت", callback_data="taraz_design")],
    ])
    
    await query.message.edit_text(
        text,
        parse_mode="Markdown",
        reply_markup=keyboard
    )


async def start_lucky_test_callback(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """شروع تست شانسی از طریق callback."""
    query = update.callback_query
    await query.answer()
    
    user_id = update.effective_user.id
    
    db_user = await UserRepository.get_by_telegram_id(user_id)
    if not db_user or not db_user.is_registration_complete():
        await query.message.edit_text(
            "❌ لطفاً ابتدا ثبت‌نام را تکمیل کنید. دستور /start را ارسال کنید."
        )
        return
    
    question = await UserQuestionRepository.get_random_one()
    
    if not question:
        await query.message.edit_text(
            "❌ متاسفانه هنوز سوالی در مخزن تست شانسی وجود ندارد.\n"
            "شما می‌توانید با «🎨 طراحی تست» سوال طراحی کنید!",
            reply_markup=InlineKeyboardMarkup([[
                InlineKeyboardButton("🏠 منوی اصلی", callback_data="back_to_menu")
            ]])
        )
        return
    
    context.user_data["lucky_question"] = question
    
    await query.message.edit_text(
        f"🎲 *تست شانسی*\n\n"
        f"{question['text']}\n\n"
        f"*الف)* {question['option_a']}\n"
        f"*ب)* {question['option_b']}\n"
        f"*ج)* {question['option_c']}\n"
        f"*د)* {question['option_d']}",
        parse_mode="Markdown",
        reply_markup=get_lucky_test_keyboard()
    )
