# models.py - کلاس‌های Data Class برای مدل‌های داده

from dataclasses import dataclass
from datetime import datetime
from typing import Optional


@dataclass
class User:
    """مدل داده کاربر."""
    telegram_id: int
    phone_number: Optional[str] = None
    full_name: Optional[str] = None
    field: Optional[str] = None
    grade: Optional[str] = None
    province: Optional[str] = None
    is_active: bool = True
    created_at: Optional[datetime] = None
    updated_at: Optional[datetime] = None

    @classmethod
    def from_row(cls, row) -> "User":
        """ایجاد User از یک ردیف دیتابیس."""
        if row is None:
            return None
        return cls(
            telegram_id=row["telegram_id"],
            phone_number=row["phone_number"],
            full_name=row["full_name"],
            field=row["field"],
            grade=row["grade"],
            province=row["province"],
            is_active=bool(row["is_active"]),
            created_at=row["created_at"],
            updated_at=row["updated_at"],
        )

    def is_registration_complete(self) -> bool:
        """بررسی تکمیل بودن ثبت‌نام کاربر."""
        return all([
            self.phone_number,
            self.full_name,
            self.field,
            self.grade,
            self.province,
        ])


@dataclass
class Question:
    """مدل داده سوال."""
    id: int
    text: str
    option_a: str
    option_b: str
    option_c: str
    option_d: str
    correct_answer: str  # 'A', 'B', 'C', 'D'
    explanation: Optional[str] = None  # پاسخنامه تشریحی
    category: Optional[str] = None
    difficulty: str = "medium"
    is_active: bool = True
    created_by: Optional[int] = None
    created_at: Optional[datetime] = None

    @classmethod
    def from_row(cls, row) -> "Question":
        """ایجاد Question از یک ردیف دیتابیس."""
        if row is None:
            return None
        return cls(
            id=row["id"],
            text=row["text"],
            option_a=row["option_a"],
            option_b=row["option_b"],
            option_c=row["option_c"],
            option_d=row["option_d"],
            correct_answer=row["correct_answer"],
            explanation=row["explanation"],
            category=row["category"],
            difficulty=row["difficulty"],
            is_active=bool(row["is_active"]),
            created_by=row["created_by"],
            created_at=row["created_at"],
        )

    def get_options(self) -> dict:
        """برگرداندن گزینه‌ها به صورت دیکشنری."""
        return {
            "A": self.option_a,
            "B": self.option_b,
            "C": self.option_c,
            "D": self.option_d,
        }


@dataclass
class ExamResult:
    """مدل داده نتیجه آزمون."""
    id: int
    user_id: int
    total_questions: int
    correct_answers: int
    score: float
    exam_date: Optional[datetime] = None

    @classmethod
    def from_row(cls, row) -> "ExamResult":
        """ایجاد ExamResult از یک ردیف دیتابیس."""
        if row is None:
            return None
        return cls(
            id=row["id"],
            user_id=row["user_id"],
            total_questions=row["total_questions"],
            correct_answers=row["correct_answers"],
            score=row["score"],
            exam_date=row["exam_date"],
        )
