# repositories.py - کوئری‌های SQL (بدون کد تلگرامی)

from typing import Optional, List
from database import Database
from models import User, Question, ExamResult


class UserRepository:
    """Repository برای عملیات CRUD روی کاربران."""

    @staticmethod
    async def get_by_telegram_id(telegram_id: int) -> Optional[User]:
        """دریافت کاربر با telegram_id."""
        conn = await Database.get_connection()
        cursor = await conn.execute(
            "SELECT * FROM users WHERE telegram_id = ?",
            (telegram_id,)
        )
        row = await cursor.fetchone()
        return User.from_row(row)

    @staticmethod
    async def get_by_phone(phone_number: str) -> Optional[User]:
        """دریافت کاربر با شماره تلفن."""
        conn = await Database.get_connection()
        cursor = await conn.execute(
            "SELECT * FROM users WHERE phone_number = ?",
            (phone_number,)
        )
        row = await cursor.fetchone()
        return User.from_row(row)

    @staticmethod
    async def insert(telegram_id: int, phone_number: str = None) -> User:
        """ایجاد کاربر جدید."""
        conn = await Database.get_connection()
        await conn.execute(
            "INSERT OR IGNORE INTO users (telegram_id, phone_number) VALUES (?, ?)",
            (telegram_id, phone_number)
        )
        await conn.commit()
        return await UserRepository.get_by_telegram_id(telegram_id)

    @staticmethod
    async def update_phone(telegram_id: int, phone_number: str) -> User:
        """بروزرسانی شماره تلفن کاربر."""
        conn = await Database.get_connection()
        await conn.execute(
            """UPDATE users 
               SET phone_number = ?, updated_at = CURRENT_TIMESTAMP 
               WHERE telegram_id = ?""",
            (phone_number, telegram_id)
        )
        await conn.commit()
        return await UserRepository.get_by_telegram_id(telegram_id)

    @staticmethod
    async def update_profile(
        telegram_id: int,
        full_name: str = None,
        field: str = None,
        grade: str = None,
        province: str = None
    ) -> User:
        """بروزرسانی پروفایل کاربر."""
        conn = await Database.get_connection()
        
        updates = []
        params = []
        
        if full_name is not None:
            updates.append("full_name = ?")
            params.append(full_name)
        if field is not None:
            updates.append("field = ?")
            params.append(field)
        if grade is not None:
            updates.append("grade = ?")
            params.append(grade)
        if province is not None:
            updates.append("province = ?")
            params.append(province)
        
        if updates:
            updates.append("updated_at = CURRENT_TIMESTAMP")
            params.append(telegram_id)
            
            query = f"UPDATE users SET {', '.join(updates)} WHERE telegram_id = ?"
            await conn.execute(query, params)
            await conn.commit()
        
        return await UserRepository.get_by_telegram_id(telegram_id)

    @staticmethod
    async def set_active(telegram_id: int, is_active: bool) -> None:
        """فعال/غیرفعال کردن کاربر."""
        conn = await Database.get_connection()
        await conn.execute(
            "UPDATE users SET is_active = ?, updated_at = CURRENT_TIMESTAMP WHERE telegram_id = ?",
            (int(is_active), telegram_id)
        )
        await conn.commit()

    @staticmethod
    async def get_all() -> List[User]:
        """دریافت لیست همه کاربران."""
        conn = await Database.get_connection()
        async with conn.execute(
            "SELECT telegram_id, phone_number, full_name, field, grade, province, is_active, created_at, updated_at FROM users WHERE is_active = 1"
        ) as cursor:
            rows = await cursor.fetchall()
            return [User(
                telegram_id=row[0],
                phone_number=row[1],
                full_name=row[2],
                field=row[3],
                grade=row[4],
                province=row[5],
                is_active=row[6],
                created_at=row[7],
                updated_at=row[8]
            ) for row in rows]




class QuestionRepository:
    """Repository برای عملیات روی سوالات."""

    @staticmethod
    async def get_by_id(question_id: int) -> Optional[Question]:
        """دریافت سوال با شناسه."""
        conn = await Database.get_connection()
        cursor = await conn.execute(
            "SELECT * FROM questions WHERE id = ? AND is_active = 1",
            (question_id,)
        )
        row = await cursor.fetchone()
        return Question.from_row(row)

    @staticmethod
    async def get_random_questions(limit: int = 10, category: str = None) -> List[Question]:
        """دریافت سوالات تصادفی."""
        conn = await Database.get_connection()
        
        if category:
            cursor = await conn.execute(
                """SELECT * FROM questions 
                   WHERE is_active = 1 AND category = ?
                   ORDER BY RANDOM() LIMIT ?""",
                (category, limit)
            )
        else:
            cursor = await conn.execute(
                """SELECT * FROM questions 
                   WHERE is_active = 1
                   ORDER BY RANDOM() LIMIT ?""",
                (limit,)
            )
        
        rows = await cursor.fetchall()
        return [Question.from_row(row) for row in rows]

    @staticmethod
    async def get_all_active() -> List[Question]:
        """دریافت همه سوالات فعال."""
        conn = await Database.get_connection()
        cursor = await conn.execute(
            "SELECT * FROM questions WHERE is_active = 1"
        )
        rows = await cursor.fetchall()
        return [Question.from_row(row) for row in rows]

    @staticmethod
    async def insert(
        text: str,
        option_a: str,
        option_b: str,
        option_c: str,
        option_d: str,
        correct_answer: str,
        explanation: str = None,
        category: str = None,
        difficulty: str = "medium",
        created_by: int = None
    ) -> int:
        """افزودن سوال جدید و برگرداندن شناسه."""
        conn = await Database.get_connection()
        cursor = await conn.execute(
            """INSERT INTO questions 
               (text, option_a, option_b, option_c, option_d, correct_answer, explanation, category, difficulty, created_by)
               VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)""",
            (text, option_a, option_b, option_c, option_d, correct_answer, explanation, category, difficulty, created_by)
        )
        await conn.commit()
        return cursor.lastrowid

    @staticmethod
    async def get_count() -> int:
        """دریافت تعداد کل سوالات فعال."""
        conn = await Database.get_connection()
        cursor = await conn.execute(
            "SELECT COUNT(*) as count FROM questions WHERE is_active = 1"
        )
        row = await cursor.fetchone()
        return row["count"] if row else 0

    @staticmethod
    async def delete(question_id: int) -> bool:
        """حذف سوال (غیرفعال کردن)."""
        conn = await Database.get_connection()
        await conn.execute(
            "UPDATE questions SET is_active = 0 WHERE id = ?",
            (question_id,)
        )
        await conn.commit()
        return True


class ExamResultRepository:
    """Repository برای ذخیره نتایج آزمون."""

    @staticmethod
    async def insert(user_id: int, total_questions: int, correct_answers: int) -> int:
        """ذخیره نتیجه آزمون."""
        score = (correct_answers / total_questions) * 100 if total_questions > 0 else 0
        conn = await Database.get_connection()
        cursor = await conn.execute(
            """INSERT INTO exam_results 
               (user_id, total_questions, correct_answers, score)
               VALUES (?, ?, ?, ?)""",
            (user_id, total_questions, correct_answers, score)
        )
        await conn.commit()
        return cursor.lastrowid

    @staticmethod
    async def get_user_results(user_id: int, limit: int = 10) -> List[ExamResult]:
        """دریافت آخرین نتایج آزمون کاربر."""
        conn = await Database.get_connection()
        cursor = await conn.execute(
            """SELECT * FROM exam_results 
               WHERE user_id = ?
               ORDER BY exam_date DESC LIMIT ?""",
            (user_id, limit)
        )
        rows = await cursor.fetchall()
        return [ExamResult.from_row(row) for row in rows]


class PendingQuestionRepository:
    """Repository برای سوالات در انتظار تایید."""

    @staticmethod
    async def insert(
        text: str,
        option_a: str,
        option_b: str,
        option_c: str,
        option_d: str,
        correct_answer: str,
        explanation: str,
        submitted_by: int
    ) -> int:
        """افزودن سوال در انتظار تایید."""
        conn = await Database.get_connection()
        cursor = await conn.execute(
            """INSERT INTO pending_questions 
               (text, option_a, option_b, option_c, option_d, correct_answer, explanation, submitted_by)
               VALUES (?, ?, ?, ?, ?, ?, ?, ?)""",
            (text, option_a, option_b, option_c, option_d, correct_answer, explanation, submitted_by)
        )
        await conn.commit()
        return cursor.lastrowid

    @staticmethod
    async def get_pending() -> list:
        """دریافت سوالات در انتظار تایید."""
        conn = await Database.get_connection()
        cursor = await conn.execute(
            """SELECT pq.*, u.full_name as submitter_name 
               FROM pending_questions pq
               LEFT JOIN users u ON pq.submitted_by = u.telegram_id
               WHERE pq.status = 'pending'
               ORDER BY pq.submitted_at ASC"""
        )
        rows = await cursor.fetchall()
        return [dict(row) for row in rows]

    @staticmethod
    async def get_by_id(question_id: int) -> dict:
        """دریافت سوال با شناسه."""
        conn = await Database.get_connection()
        cursor = await conn.execute(
            "SELECT * FROM pending_questions WHERE id = ?",
            (question_id,)
        )
        row = await cursor.fetchone()
        return dict(row) if row else None

    @staticmethod
    async def approve(question_id: int, reviewed_by: int) -> bool:
        """تایید سوال و انتقال به مخزن B."""
        conn = await Database.get_connection()
        
        # دریافت سوال
        cursor = await conn.execute(
            "SELECT * FROM pending_questions WHERE id = ?",
            (question_id,)
        )
        row = await cursor.fetchone()
        if not row:
            return False
        
        # انتقال به جدول user_questions
        await conn.execute(
            """INSERT INTO user_questions 
               (text, option_a, option_b, option_c, option_d, correct_answer, explanation, submitted_by)
               VALUES (?, ?, ?, ?, ?, ?, ?, ?)""",
            (row["text"], row["option_a"], row["option_b"], row["option_c"], 
             row["option_d"], row["correct_answer"], row["explanation"], row["submitted_by"])
        )
        
        # بروزرسانی وضعیت
        await conn.execute(
            """UPDATE pending_questions 
               SET status = 'approved', reviewed_by = ?, reviewed_at = CURRENT_TIMESTAMP
               WHERE id = ?""",
            (reviewed_by, question_id)
        )
        await conn.commit()
        return True

    @staticmethod
    async def reject(question_id: int, reviewed_by: int) -> bool:
        """رد سوال."""
        conn = await Database.get_connection()
        await conn.execute(
            """UPDATE pending_questions 
               SET status = 'rejected', reviewed_by = ?, reviewed_at = CURRENT_TIMESTAMP
               WHERE id = ?""",
            (reviewed_by, question_id)
        )
        await conn.commit()
        return True

    @staticmethod
    async def get_pending_count() -> int:
        """دریافت تعداد سوالات در انتظار."""
        conn = await Database.get_connection()
        cursor = await conn.execute(
            "SELECT COUNT(*) as count FROM pending_questions WHERE status = 'pending'"
        )
        row = await cursor.fetchone()
        return row["count"] if row else 0

    @staticmethod
    async def get_by_user(user_id: int) -> list:
        """دریافت سوالات در انتظار یک کاربر."""
        conn = await Database.get_connection()
        cursor = await conn.execute(
            """SELECT * FROM pending_questions 
               WHERE submitted_by = ? AND status = 'pending'
               ORDER BY submitted_at DESC""",
            (user_id,)
        )
        rows = await cursor.fetchall()
        return [dict(row) for row in rows]


class RatingRepository:
    """Repository برای سیستم امتیازدهی."""

    @staticmethod
    async def add_rating(user_id: int, question_id: int, score: int) -> bool:
        """ثبت امتیاز جدید."""
        conn = await Database.get_connection()
        try:
            # 1. ثبت امتیاز در جدول ratings
            await conn.execute(
                """INSERT INTO test_ratings (user_id, question_id, score) 
                   VALUES (?, ?, ?)""",
                (user_id, question_id, score)
            )
            
            # 2. آپدیت میانگین در جدول user_questions
            # ابتدا مقادیر فعلی را می‌گیریم
            cursor = await conn.execute(
                "SELECT rating_sum, rating_count FROM user_questions WHERE id = ?",
                (question_id,)
            )
            row = await cursor.fetchone()
            if row:
                current_sum = row["rating_sum"] or 0
                current_count = row["rating_count"] or 0
                
                new_sum = current_sum + score
                new_count = current_count + 1
                
                await conn.execute(
                    """UPDATE user_questions 
                       SET rating_sum = ?, rating_count = ? 
                       WHERE id = ?""",
                    (new_sum, new_count, question_id)
                )
            
            await conn.commit()
            return True
        except Exception as e:
            # احتمالا قبلا رای داده است
            return False

    @staticmethod
    async def get_user_rating(user_id: int, question_id: int) -> int:
        """دریافت امتیاز کاربر به یک سوال (اگر رای داده باشد)."""
        conn = await Database.get_connection()
        cursor = await conn.execute(
            "SELECT score FROM test_ratings WHERE user_id = ? AND question_id = ?",
            (user_id, question_id)
        )
        row = await cursor.fetchone()
        return row["score"] if row else 0


class UserQuestionRepository:
    """Repository برای سوالات تایید شده کاربران (مخزن B)."""

    @staticmethod
    async def get_random_one(user_id: int = None):
        """دریافت یک سوال تصادفی از مخزن کاربران (همراه با اطلاعات امتیاز).
        اگر user_id داده شود، سوالاتی که کاربر قبلا دیده را exclude می‌کنیم.
        """
        conn = await Database.get_connection()
        
        if user_id:
            # سوالاتی که کاربر قبلا دیده را exclude کن
            cursor = await conn.execute(
                """SELECT uq.*, u.full_name as maker_name 
                   FROM user_questions uq
                   LEFT JOIN users u ON uq.submitted_by = u.telegram_id
                   WHERE uq.is_active = 1
                   AND uq.id NOT IN (
                       SELECT question_id FROM user_lucky_history WHERE user_id = ?
                   )
                   ORDER BY RANDOM() LIMIT 1""",
                (user_id,)
            )
        else:
            cursor = await conn.execute(
                """SELECT uq.*, u.full_name as maker_name 
                   FROM user_questions uq
                   LEFT JOIN users u ON uq.submitted_by = u.telegram_id
                   WHERE uq.is_active = 1
                   ORDER BY RANDOM() LIMIT 1"""
            )
        row = await cursor.fetchone()
        return dict(row) if row else None

    @staticmethod
    async def mark_seen(user_id: int, question_id: int) -> bool:
        """ثبت اینکه کاربر این سوال را دیده."""
        conn = await Database.get_connection()
        try:
            await conn.execute(
                "INSERT OR IGNORE INTO user_lucky_history (user_id, question_id) VALUES (?, ?)",
                (user_id, question_id)
            )
            await conn.commit()
            return True
        except Exception:
            return False

    @staticmethod
    async def get_count() -> int:
        """دریافت تعداد سوالات در مخزن B."""
        conn = await Database.get_connection()
        cursor = await conn.execute(
            "SELECT COUNT(*) as count FROM user_questions WHERE is_active = 1"
        )
        row = await cursor.fetchone()
        return row["count"] if row else 0

    @staticmethod
    async def get_by_user(user_id: int) -> list:
        """دریافت سوالات تایید شده یک کاربر."""
        conn = await Database.get_connection()
        cursor = await conn.execute(
            """SELECT * FROM user_questions 
               WHERE submitted_by = ? AND is_active = 1
               ORDER BY created_at DESC""",
            (user_id,)
        )
        rows = await cursor.fetchall()
        return [dict(row) for row in rows]


class PreexamRepository:
    """Repository برای مدیریت پیش‌آزمون‌ها."""

    @staticmethod
    async def create(title: str, description: str = None, created_by: int = None) -> int:
        """ایجاد پیش‌آزمون جدید."""
        conn = await Database.get_connection()
        cursor = await conn.execute(
            """INSERT INTO preexams (title, description, created_by)
               VALUES (?, ?, ?)""",
            (title, description, created_by)
        )
        await conn.commit()
        return cursor.lastrowid

    @staticmethod
    async def get_all_active() -> list:
        """دریافت همه پیش‌آزمون‌های فعال."""
        conn = await Database.get_connection()
        cursor = await conn.execute(
            """SELECT * FROM preexams WHERE is_active = 1 ORDER BY order_num, id"""
        )
        rows = await cursor.fetchall()
        return [dict(row) for row in rows]

    @staticmethod
    async def get_by_id(preexam_id: int) -> dict:
        """دریافت پیش‌آزمون با شناسه."""
        conn = await Database.get_connection()
        cursor = await conn.execute(
            "SELECT * FROM preexams WHERE id = ?",
            (preexam_id,)
        )
        row = await cursor.fetchone()
        return dict(row) if row else None

    @staticmethod
    async def add_question(preexam_id: int, question_id: int, order: int = 0) -> bool:
        """افزودن سوال به پیش‌آزمون."""
        conn = await Database.get_connection()
        try:
            await conn.execute(
                """INSERT INTO preexam_questions (preexam_id, question_id, question_order)
                   VALUES (?, ?, ?)""",
                (preexam_id, question_id, order)
            )
            await conn.commit()
            return True
        except:
            return False

    @staticmethod
    async def remove_question(preexam_id: int, question_id: int) -> bool:
        """حذف سوال از پیش‌آزمون."""
        conn = await Database.get_connection()
        await conn.execute(
            "DELETE FROM preexam_questions WHERE preexam_id = ? AND question_id = ?",
            (preexam_id, question_id)
        )
        await conn.commit()
        return True

    @staticmethod
    async def get_questions(preexam_id: int) -> list:
        """دریافت سوالات یک پیش‌آزمون."""
        conn = await Database.get_connection()
        cursor = await conn.execute(
            """SELECT q.* FROM questions q
               INNER JOIN preexam_questions pq ON q.id = pq.question_id
               WHERE pq.preexam_id = ? AND q.is_active = 1
               ORDER BY pq.question_order, pq.id""",
            (preexam_id,)
        )
        rows = await cursor.fetchall()
        return [dict(row) for row in rows]

    @staticmethod
    async def get_question_count(preexam_id: int) -> int:
        """دریافت تعداد سوالات پیش‌آزمون."""
        conn = await Database.get_connection()
        cursor = await conn.execute(
            "SELECT COUNT(*) as count FROM preexam_questions WHERE preexam_id = ?",
            (preexam_id,)
        )
        row = await cursor.fetchone()
        return row["count"] if row else 0

    @staticmethod
    async def delete(preexam_id: int) -> bool:
        """حذف پیش‌آزمون (غیرفعال کردن)."""
        conn = await Database.get_connection()
        await conn.execute(
            "UPDATE preexams SET is_active = 0 WHERE id = ?",
            (preexam_id,)
        )
        await conn.commit()
        return True


class PreexamResultRepository:
    """Repository برای نتایج پیش‌آزمون."""

    @staticmethod
    async def save_result(
        user_id: int,
        preexam_id: int,
        total: int,
        correct: int,
        wrong: int,
        blank: int,
        score_simple: float,
        score_negative: float,
        answers_json: str = None
    ) -> int:
        """ذخیره نتیجه پیش‌آزمون."""
        conn = await Database.get_connection()
        cursor = await conn.execute(
            """INSERT INTO preexam_results 
               (user_id, preexam_id, total_questions, correct_answers, wrong_answers, 
                blank_answers, score_simple, score_negative, answers_json)
               VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)""",
            (user_id, preexam_id, total, correct, wrong, blank, score_simple, score_negative, answers_json)
        )
        await conn.commit()
        return cursor.lastrowid

    @staticmethod
    async def get_user_result(user_id: int, preexam_id: int) -> dict:
        """دریافت آخرین نتیجه کاربر برای یک پیش‌آزمون."""
        conn = await Database.get_connection()
        cursor = await conn.execute(
            """SELECT * FROM preexam_results 
               WHERE user_id = ? AND preexam_id = ?
               ORDER BY exam_date DESC LIMIT 1""",
            (user_id, preexam_id)
        )
        row = await cursor.fetchone()
        return dict(row) if row else None

    @staticmethod
    async def get_user_all_results(user_id: int) -> list:
        """دریافت همه نتایج پیش‌آزمون کاربر."""
        conn = await Database.get_connection()
        cursor = await conn.execute(
            """SELECT pr.*, p.title as preexam_title 
               FROM preexam_results pr
               INNER JOIN preexams p ON pr.preexam_id = p.id
               WHERE pr.user_id = ?
               ORDER BY pr.exam_date DESC""",
            (user_id,)
        )
        rows = await cursor.fetchall()
        return [dict(row) for row in rows]

    @staticmethod
    async def has_taken_exam(user_id: int, preexam_id: int) -> bool:
        """بررسی اینکه آیا کاربر در این پیش‌آزمون شرکت کرده است."""
        conn = await Database.get_connection()
        cursor = await conn.execute(
            "SELECT COUNT(*) as count FROM preexam_results WHERE user_id = ? AND preexam_id = ?",
            (user_id, preexam_id)
        )

        return row["count"] > 0 if row else False


class ServiceRequestRepository:
    """Repository برای درخواست‌های خدمات."""

    @staticmethod
    async def insert(user_id: int, phone: str, service_type: str, receipt_file_id: str = None) -> int:
        """ذخیره درخواست جدید."""
        conn = await Database.get_connection()
        
        # چک وجود جدول درخواست‌ها
        await conn.execute("""
            CREATE TABLE IF NOT EXISTS service_requests (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                user_id INTEGER NOT NULL,
                phone TEXT NOT NULL,
                service_type TEXT NOT NULL,
                status TEXT DEFAULT 'pending',
                receipt_file_id TEXT,
                is_approved INTEGER DEFAULT 0,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                FOREIGN KEY (user_id) REFERENCES users(telegram_id)
            )
        """)
        
        # مایگریشن برای جدول درخواست‌ها
        try:
            await conn.execute("ALTER TABLE service_requests ADD COLUMN receipt_file_id TEXT")
        except Exception:
            pass
            
        try:
            await conn.execute("ALTER TABLE service_requests ADD COLUMN is_approved INTEGER DEFAULT 0")
        except Exception:
            pass
        
        cursor = await conn.execute(
            """INSERT INTO service_requests (user_id, phone, service_type, receipt_file_id)
               VALUES (?, ?, ?, ?)""",
            (user_id, phone, service_type, receipt_file_id)
        )
        await conn.commit()
        return cursor.lastrowid

    @staticmethod
    async def get_pending() -> list:
        """دریافت درخواست‌های در انتظار (همراه با اطلاعات کاربر)."""
        conn = await Database.get_connection()
        cursor = await conn.execute(
            """SELECT sr.*, u.full_name, u.field, u.grade, u.province
               FROM service_requests sr
               LEFT JOIN users u ON sr.user_id = u.telegram_id
               WHERE sr.status = 'pending'
               ORDER BY sr.created_at ASC"""
        )
        rows = await cursor.fetchall()
        return [dict(row) for row in rows]

    @staticmethod
    async def get_by_id(request_id: int) -> dict:
        """دریافت درخواست با شناسه."""
        conn = await Database.get_connection()
        cursor = await conn.execute(
            """SELECT sr.*, u.full_name, u.field, u.grade, u.province
               FROM service_requests sr
               LEFT JOIN users u ON sr.user_id = u.telegram_id
               WHERE sr.id = ?""",
            (request_id,)
        )
        row = await cursor.fetchone()
        return dict(row) if row else None

    @staticmethod
    async def update_status(request_id: int, status: str, is_approved: int) -> bool:
        """بروزرسانی وضعیت درخواست."""
        conn = await Database.get_connection()
        await conn.execute(
            "UPDATE service_requests SET status = ?, is_approved = ? WHERE id = ?",
            (status, is_approved, request_id)
        )
        await conn.commit()
        return True

    @staticmethod
    async def delete(request_id: int) -> bool:
        """حذف درخواست."""
        conn = await Database.get_connection()
        await conn.execute("DELETE FROM service_requests WHERE id = ?", (request_id,))
        await conn.commit()
        return True

    @staticmethod
    async def has_pending_request(user_id: int) -> bool:
        """بررسی وجود درخواست در انتظار برای کاربر."""
        conn = await Database.get_connection()
        cursor = await conn.execute(
            "SELECT COUNT(*) as count FROM service_requests WHERE user_id = ? AND status = 'pending'",
            (user_id,)
        )
        row = await cursor.fetchone()
        return row["count"] > 0 if row else False

    @staticmethod
    async def get_approved() -> list:
        """دریافت درخواست‌های تایید شده (همراه با اطلاعات کاربر)."""
        conn = await Database.get_connection()
        cursor = await conn.execute(
            """SELECT sr.*, u.full_name, u.field, u.grade, u.province
               FROM service_requests sr
               LEFT JOIN users u ON sr.user_id = u.telegram_id
               WHERE sr.status = 'approved'
               ORDER BY sr.created_at DESC"""
        )
        rows = await cursor.fetchall()
        return [dict(row) for row in rows]


class SettingsRepository:
    """Repository برای تنظیمات."""

    @staticmethod
    async def get(key: str, default: str = None) -> str:
        """دریافت مقدار یک تنظیم."""
        conn = await Database.get_connection()
        cursor = await conn.execute(
            "SELECT value FROM settings WHERE key = ?",
            (key,)
        )
        row = await cursor.fetchone()
        return row["value"] if row else default

    @staticmethod
    async def set(key: str, value: str) -> bool:
        """ذخیره مقدار یک تنظیم."""
        conn = await Database.get_connection()
        await conn.execute(
            "INSERT OR REPLACE INTO settings (key, value, updated_at) VALUES (?, ?, CURRENT_TIMESTAMP)",
            (key, value)
        )
        await conn.commit()
        return True

    @staticmethod
    async def get_exam_price() -> int:
        """دریافت قیمت پکیج آزمون."""
        price = await SettingsRepository.get("exam_price", "800000")
        return int(price)

    @staticmethod
    async def get_consult_price() -> int:
        """دریافت قیمت مشاوره."""
        price = await SettingsRepository.get("consult_price", "850000")
        return int(price)
