# services.py - لاجیک اصلی برنامه

import re
from telegram import Bot
from telegram.error import TelegramError
from config import REQUIRED_CHANNEL_ID


class MembershipService:
    """سرویس بررسی عضویت کانال."""

    @staticmethod
    async def check_channel_membership(bot: Bot, user_id: int) -> bool:
        """
        بررسی عضویت کاربر در کانال اجباری.
        
        Returns:
            True اگر کاربر عضو باشد، False در غیر این صورت.
        """
        try:
            member = await bot.get_chat_member(
                chat_id=REQUIRED_CHANNEL_ID,
                user_id=user_id
            )
            # وضعیت‌های قابل قبول برای عضویت
            allowed_statuses = ["member", "administrator", "creator"]
            return member.status in allowed_statuses
        except TelegramError:
            # اگر ربات دسترسی نداشت یا خطایی رخ داد
            return False


class ValidationService:
    """سرویس اعتبارسنجی داده‌ها."""

    @staticmethod
    def validate_phone(phone_number: str) -> bool:
        """
        اعتبارسنجی فرمت شماره تلفن ایرانی.
        
        پشتیبانی از فرمت‌های:
        - 09123456789
        - +989123456789
        - 989123456789
        """
        if not phone_number:
            return False
        
        # حذف فضاهای خالی و کاراکترهای اضافه
        phone = phone_number.strip().replace(" ", "").replace("-", "")
        
        # الگوهای قابل قبول
        patterns = [
            r"^09\d{9}$",           # 09123456789
            r"^\+989\d{9}$",        # +989123456789
            r"^989\d{9}$",          # 989123456789
        ]
        
        return any(re.match(pattern, phone) for pattern in patterns)

    @staticmethod
    def normalize_phone(phone_number: str) -> str:
        """
        نرمال‌سازی شماره تلفن به فرمت 09XXXXXXXXX
        """
        phone = phone_number.strip().replace(" ", "").replace("-", "")
        
        if phone.startswith("+98"):
            phone = "0" + phone[3:]
        elif phone.startswith("98"):
            phone = "0" + phone[2:]
        
        return phone

    @staticmethod
    def validate_name(name: str) -> bool:
        """اعتبارسنجی نام (حداقل ۲ کاراکتر فارسی یا انگلیسی)."""
        if not name or len(name.strip()) < 2:
            return False
        # فقط حروف فارسی، انگلیسی و فاصله مجاز است
        pattern = r"^[\u0600-\u06FFa-zA-Z\s]+$"
        return bool(re.match(pattern, name.strip()))


class TextService:
    """سرویس متن‌های ربات (برای مدیریت متمرکز پیام‌ها)."""

    WELCOME_MESSAGE = """
🎓 *به ربات آزمون خوش آمدید!*

برای استفاده از امکانات ربات، ابتدا باید در کانال ما عضو شوید.
"""

    MEMBERSHIP_REQUIRED = """
⚠️ *عضویت در کانال الزامی است*

لطفاً ابتدا در کانال ما عضو شوید و سپس دکمه «✅ عضو شدم» را بزنید.
"""

    PHONE_REQUEST = """
📱 *ارسال شماره تماس*

لطفاً شماره تماس خود را با استفاده از دکمه زیر ارسال کنید.
"""

    REGISTRATION_NAME = """
📝 *ثبت‌نام - مرحله ۱ از ۴*

لطفاً نام و نام خانوادگی خود را وارد کنید:
"""

    REGISTRATION_FIELD = """
📚 *ثبت‌نام - مرحله ۲ از ۴*

رشته تحصیلی خود را انتخاب کنید:
"""

    REGISTRATION_GRADE = """
🎒 *ثبت‌نام - مرحله ۳ از ۴*

پایه تحصیلی خود را انتخاب کنید:
"""

    REGISTRATION_PROVINCE = """
🏠 *ثبت‌نام - مرحله ۴ از ۴*

استان محل سکونت خود را انتخاب کنید:
"""

    REGISTRATION_COMPLETE = """
✅ *ثبت‌نام با موفقیت انجام شد!*

اطلاعات شما:
👤 نام: {full_name}
📚 رشته: {field}
🎒 پایه: {grade}
🏠 استان: {province}

اکنون می‌توانید از منوی اصلی استفاده کنید.
"""

    MAIN_MENU = """
🏠 *منوی اصلی*

از گزینه‌های زیر انتخاب کنید:
"""

    @staticmethod
    def format_registration_complete(user) -> str:
        """فرمت پیام تکمیل ثبت‌نام."""
        return TextService.REGISTRATION_COMPLETE.format(
            full_name=user.full_name,
            field=user.field,
            grade=user.grade,
            province=user.province,
        )
